#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#define PORT                        10000           //PORT ADDRESS
#define FITS_HEADER_SIZE            16384           //FITS HEADER SIZE
#define ACKNOWLEDGE_SIZE            100

#define END_MSG_CHAR                '\n'
#define END_MSG_STRING              "\n"

#define ACQUISITION_CMD             "ACQ"     
#define END_ACQUISITION_CMD         "END_ACQ" 

#define ACKNOLEDGE_RECEPTION        "DATA RECEIVED" 
#define ACKNOLEDGE_EXECUTION_OK     "OK"      
#define ACKNOLEDGE_EXECUTION_ERR    "ERROR"   

#define FILENAME_KEYW               "FILENAME"
#define TIME_EXPO_KEYW              "EXPTIME"
#define OBS_MODE                    "OBS_MODE"
#define CAMERA_NUMBER               "NOCAM"

#define SUCCESS                     1
#define FAILURE                     0

#define TRUE                        1
#define FALSE                       0

#define MASTER_CAMERA_NUMBER        2
#define SLAVE_CAMERA_NUMBER         1

#define ACQUISITION_PATH            "/home/lefort/SoftDpsm/pc2/"

#define SLAVE_HOST                  "hydre"

#define MASTER                      FALSE
#define DEBUG_MODE                  TRUE

char    fitsFileName[256]   = "";
char    acquisitionMode[80] = "";
float   exposureTime        = 0;
int     cameraNumber        = MASTER_CAMERA_NUMBER;
int     fitsHeaderLoaded    = FALSE;
int     fitsFileReady       = FALSE;
FILE    *fitsFileDescriptor;
char    *mainFitsHeader     = NULL;

//This function look for <keyword> in <fitsHeader> and return its
//value in <value>
//WARNING: <value> must be big enough to store the result.
int fitsFindInHeader(char *fitsHeader, char *keyWord, char *value)
{
    char *position;
    int  i;
    int  j;
    
    //Look for the keyword
    //If is not present 
    if ((position = strstr(fitsHeader,keyWord)) == NULL)
    {
        //Retrun an error
        printf("ERROR:\t\tKEYWORD NOT FOUND IN THE HEADER\n");
        return FAILURE;
    }
    
    //Find the equal sign
    //If is not present
    if ((char)position[8] != '=')
    {
        //Retrun an error
        printf("ERROR:\t\tINVALID fITS STRUCTURE\n");
        return FAILURE;
    }

    //Skip all the spaces
    i = 1;
    while (((char)position[8+i] == ' ') && (i <= 70))
    {
        i++;
    }

    //If There is only spaces
    if ( i == 70)
    {
        //return an error
        printf("ERROR:\t\tKEYWORD NOT FILLED IN THE HEADER\n");
        return FAILURE;
    }
    
    //If the keyword value is a string
    if((char)position[8+i] == '\'')
    {
        if (DEBUG_MODE == TRUE)
        printf("INFORMATION:\tKeyword '%s' is a string: '", keyWord);
        //Retrieve string chars
        j = 1;
        while((8+i+j <= 79) && (position[8+i+j] != '\''))
        {
            *value = position[8+i+j];
            if (DEBUG_MODE == TRUE)
            printf("%c",position[8+i+j]);
            j++;
            value++;
            
        }
        if (DEBUG_MODE == TRUE)
        printf("'\n");
        //if the last char is a single quote
        if(position[8+i+j] == '\'')
        {
            //return SUCCESS
            return SUCCESS;
        }
        //else
        else
        {
            //Return an error
            printf("ERROR:\t\tINVALID KEYWORD VALUE\n");
            return FAILURE;
        }
    }
    //If the keyword is a number
    else
    {
        if (DEBUG_MODE == TRUE)
        printf("INFORMATION:\tKeyword '%s' is a number : '", keyWord);
        //Retrieve number digits
        j = 0;
        while((8+i+j <= 79) && (isdigit((int)position[8+i+j]) != 0))
        {
            *value = position[8+i+j];
            if (DEBUG_MODE == TRUE)
            printf("%c",position[8+i+j]);
            j++;
            value++;
        }

        //Verify if there is something behind the coma
        if (position[8+i+j] == '.')
        {
            *value = position[8+i+j];
            value++;
            if (DEBUG_MODE == TRUE)
            printf("%c",position[8+i+j]);
            j++;

            //Copy all the data behind the coma
            while((8+i+j <= 79) && (isdigit((int)position[8+i+j]) != 0))
            {
                *value = position[8+i+j];
                if (DEBUG_MODE == TRUE)
                printf("%c",position[8+i+j]);
                j++;
                value++;
            }
        } 
        if (DEBUG_MODE == TRUE)
        printf("'\n");
        return SUCCESS;
    }
}

//This function send a <buffer> to the DPSM PC client
int sendToSlave(char *buffer)
{
    char    ack[ACKNOWLEDGE_SIZE];    //FITS HEADER
    int     sd;                       //SOCKET DESCRIPTOR
    struct  sockaddr_in pin;          //BORDEL POUR SOCKET
    struct  hostent     *hp;
    int     received;
    int     i, k;
    int     position = 0;
    int     foundEndMsg = FALSE;

    //Find the server
    if ((hp = gethostbyname(SLAVE_HOST)) == 0)
    {
        perror("FATAL ERROR:\t\t");
        return(FAILURE);
    }

    //FILL the socket structure with host information
    bzero(&pin, sizeof(pin));
    pin.sin_family = AF_INET;
    pin.sin_addr = *(struct in_addr *)(hp->h_addr);
    pin.sin_port = htons(PORT);

    //Get an internet socket 
    if ((sd = socket(AF_INET, SOCK_STREAM, 0)) == -1)
    {
        perror("FATAL ERROR:\t\t");
        close(sd);
        exit(FAILURE);
    }

    //Connect to PORT on HOST
    if (connect(sd, (struct sockaddr *) &pin,sizeof(pin)) == -1)
    {
        perror("FATAL ERROR:\t\t");
        close(sd);
        exit(FAILURE);
    }
    if (DEBUG_MODE == TRUE)
    printf("INFORMATION:\tConnection with slave established\n");

    //Send a message
    if (send(sd, buffer, strlen(buffer), 0) == -1)
    {
        perror("FATAL ERROR:\t\t");
        close(sd);
        exit(FAILURE);
    }

    //Send message End
    if (send(sd, END_MSG_STRING, 1, 0) == -1)
    {
        perror("FATAL ERROR:\t\t");
        close(sd);
        exit(FAILURE);
    }

    if (DEBUG_MODE == TRUE)
    {
        printf("INFORMATION:\tMessage to slave sent(%d char)\n",
               strlen(buffer));
    }

    
    //Check if the transmission is ended
    while (foundEndMsg == FALSE)
    {
        //Initialize reception buffer
        memset(&ack[0],NULL,ACKNOWLEDGE_SIZE*sizeof(char));

        //Get the data reception acknowledge from the client
        if((received = recv(sd, ack, ACKNOWLEDGE_SIZE, MSG_PEEK)) == -1)
        {
            perror("FATAL ERROR:\t\t");
            close(sd);
            exit(FAILURE);
        }
        i=0;
        while((i < received) && (foundEndMsg == FALSE))
        {   
            if (ack[i] == END_MSG_CHAR)
            {
                position = i;
                foundEndMsg = TRUE;
            }
            i++;
        }
    }
    
    //Initialize reception buffer
    memset(ack,0,ACKNOWLEDGE_SIZE*sizeof(char));

    //Get the data reception acknowledge from the client
    if((recv(sd, ack, position+1, 0)) == -1)
    {
        perror("FATAL ERROR:\t\t");
        close(sd);
        exit(FAILURE);
    }
    ack[position] = NULL;
    

    if (DEBUG_MODE == TRUE)
    printf("INFORMATION:\tReception of the slave acknowledge : '%s' \n", ack);
    
    //Check the acknowledge message
    if(strcmp(ack, ACKNOLEDGE_RECEPTION) != 0)
    {
        printf("ERROR:\t\tTHE PC CLIENT REPLY A STRANGE MESSAGE: '%s' %d %d\n",
               ack, position, received);
        for (k = 0;k < position+1;k++) printf("%x ",ack[k]);
        printf("\n");
        close(sd);
        return FAILURE;
    }
    
    foundEndMsg = FALSE;
    //Check if the transmission is ended
    while (foundEndMsg == FALSE)
    {
        //Initialize reception buffer
        memset(ack,0,ACKNOWLEDGE_SIZE*sizeof(char));

        //Get the data reception acknowledge from the client
        if((received = recv(sd, ack, ACKNOWLEDGE_SIZE, MSG_PEEK)) == -1)
        {
            perror("FATAL ERROR:\t\t");
            close(sd);
            exit(FAILURE);
        }

        i=0;
        while((i < received) && (foundEndMsg == FALSE))
        {   
            if (ack[i] == END_MSG_CHAR)
            {
                position = i;
                foundEndMsg = TRUE;
            }
            i++;
        }
    }
    //Initialize reception buffer
    memset(ack,0,ACKNOWLEDGE_SIZE*sizeof(char));

    //Get the data completion acknowledge from the client
    if((recv(sd, ack, position+1, 0)) == -1)
    {
        perror("FATAL ERROR:\t\t");
        close(sd);
        exit(FAILURE);
    }
    ack[position] = NULL;

    if (DEBUG_MODE == TRUE)
    printf("INFORMATION:\tClient result acknowledge : '%s' \n", ack);

    //Check the acknowledge message
    if (strcmp(ack,ACKNOLEDGE_EXECUTION_ERR) == 0)
    {
        printf("ERROR:\t\tTHE DPSM PC CLIENT HAS RETURNED AN ERROR '%s'\n",
               ack);
        close(sd);
        return(FAILURE);
    }

    if (strcmp(ack, ACKNOLEDGE_EXECUTION_OK) == 0)
    {
        close(sd);
        return(SUCCESS);
    }
    else
    {
        printf("ERROR:\t\tTHE DPSM PC CLIENT REPLY A NOT UNDERSTANDABLE MESSAGE: '%s'\n", ack);
        close(sd);
        return(FAILURE);
    }
}

//This function open a file using the fits header parameters
int initAcquisition()
{
    char path[1024] = "";
    int nbOfItem = 0;

    //if we are on the right PC
    if (((MASTER == TRUE) && (cameraNumber == MASTER_CAMERA_NUMBER))
        || 
        ((MASTER == FALSE) && (cameraNumber == SLAVE_CAMERA_NUMBER)))
    {

        //Create full file name
        strcpy(path, ACQUISITION_PATH);
        strcat(path,fitsFileName);
        if (DEBUG_MODE == TRUE)
            printf("INFORMATION:\tCreating the fits file: '%s'\n", path);

        //Open the file
        fitsFileDescriptor = fopen(path, "w");
        if (fitsFileDescriptor == NULL)
        {
            printf("ERROR: IMPOSSIBLE TO OPEN THE FILE: %s\n", path);
            return FAILURE;
        }

        //Write the header
        if (DEBUG_MODE == TRUE)
            printf("INFORMATION:\tWriting the FITS header\n");
        nbOfItem = fwrite(mainFitsHeader, sizeof(char), strlen(mainFitsHeader),
                          fitsFileDescriptor);
        //Check if everything has benn writed
        if (nbOfItem != strlen(mainFitsHeader))
        {
            printf("ERROR: IMPOSSIBLE TO WRITE THE FILE");
            return FAILURE;
        } 

        //Set status variable
        fitsHeaderLoaded = TRUE;
        fitsFileReady    = TRUE;
        if (DEBUG_MODE == TRUE)
            printf("INFORMATION:\tFile successfully created\n");
        return SUCCESS;
    }
    //else (if we are in the wrong PC)
    else
    {
        //Send fits header to the other PC 
        if (sendToSlave(mainFitsHeader) == SUCCESS)
        {
            //Set status variable
            fitsHeaderLoaded = TRUE;
            fitsFileReady    = TRUE;
            return SUCCESS;
        }
        else
        {
            return FAILURE;
        }
    }
}

//This funtion perform an acquisition and put the result in the fits file
int doAcquisition()
{
    if (fitsFileReady == FALSE)
    {
        printf("ERROR:\t\tTHE fITS HEADER IS NOT LOADED\n");
        return FAILURE;
    }
    if (DEBUG_MODE == TRUE)
    printf("INFORMATION:\tDoing an acquisition.....\n");
    return SUCCESS;
}

//This function close an acquisition
int doEndAcquisition()
{
    if (fitsFileReady == FALSE)
    {
        printf("ERROR:\t\tTHE fITS HEADER IS NOT LOADED\n");
        return FAILURE;
    }
    if (DEBUG_MODE == TRUE)
    printf("INFORMATION:\tClose the fits file\n");
    
    //Close file
    if (fclose(fitsFileDescriptor) != 0)
    {
        //Update status variable
        printf("ERROR:\t\tIMPOSSIBLE TO CLOSE THE FITS FILE\n");
        fitsHeaderLoaded = FALSE;
        fitsFileReady    = FALSE;
        return FAILURE;
    }
    else
    {
        if (DEBUG_MODE == TRUE)
        printf("INFORMATION:\tThe fits file has been closed\n");

        fitsHeaderLoaded = FALSE;
        fitsFileReady    = FALSE;
        return SUCCESS;
    }
}

//This function call the acquisition function or send the acquisition command to
//the other PC
int startAcquisition()
{
    if (((MASTER == TRUE) && (cameraNumber == MASTER_CAMERA_NUMBER))
        || 
        ((MASTER == FALSE) && (cameraNumber == SLAVE_CAMERA_NUMBER)))
    {
        return doAcquisition();
    }
    else
    {
        return sendToSlave(ACQUISITION_CMD);
    }
}

//This function call the end acquisition function or send the acquisition 
//command to the other PC
int endAcquisition()
{
    if (((MASTER == TRUE) && (cameraNumber == MASTER_CAMERA_NUMBER))
        || 
        ((MASTER == FALSE) && (cameraNumber == SLAVE_CAMERA_NUMBER)))
    {
        return doEndAcquisition();
    }
    else
    {
        return sendToSlave(END_ACQUISITION_CMD);
    }
}

//This function parse a command message
int commandExecution(char *command)
{
    int error;
    
    if (strcmp(command, ACQUISITION_CMD) == 0)
    {
        if (DEBUG_MODE == TRUE)
        printf("INFORMATION:\tStart an acquisition\n");
        return startAcquisition();
    }
    else
    {
        if (strcmp(command, END_ACQUISITION_CMD) == 0)
        {
            if (DEBUG_MODE == TRUE)
            printf("INFORMATION:\tEnd an acquisition\n");
            error = endAcquisition();
            fitsHeaderLoaded = FALSE;
            fitsFileReady    = FALSE;
            return error;

        }
        else
        {
            printf("ERROR:\t\tDATA ARE NEITHER FITS HEADER NOR COMMAND\n");
            return FAILURE;
        }
    }
}

//This function process a received message in order to determine if it is a fits
//header or a command
int processData(char *fitsHeader)
{
    char    tmp[80] = "";
    char    cameraNumberString[80] = "";
    
    //Check The fits header size
    if (strlen(fitsHeader) < 2880)
    {
        return commandExecution(fitsHeader);
    }

    
    //************ ICI ajouter un TAILLE / 2880 == ENTIER **********


    if (fitsHeaderLoaded == TRUE)
    {
        if (DEBUG_MODE == TRUE)
        printf("WARNING:\tA SECOND FITS HEADER HAS BEEN RECEIVED \n");
        if (endAcquisition() == FAILURE)
        {
            fitsHeaderLoaded = FALSE;
            fitsFileReady    = FALSE;
            return FAILURE;
        }
        else
        {
            fitsHeaderLoaded = FALSE;
            fitsFileReady    = FALSE;
        }
    }

    if (DEBUG_MODE == TRUE) 
    {
        printf("********************************************************\n");
        printf("******************** NEW OBSERVATION *******************\n");
        printf("********************************************************\n");
    }
    //Check FILENAME_KEYW
    if (fitsFindInHeader(fitsHeader, FILENAME_KEYW,
                         &fitsFileName[0]) == FAILURE)
    {
        return FAILURE;
    }

    if (strcmp(fitsFileName,"") == 0)
    {
        printf("ERROR:\t\tKEYWORD '%s' NOT DEFINED IN THE FITS HEADER\n",
               FILENAME_KEYW);
        return FAILURE;
    }

    //Check TIME_EXPO_KEYW
    if (fitsFindInHeader(fitsHeader, TIME_EXPO_KEYW, &tmp[0]) == FAILURE)
    {
        return FAILURE;
    }

    if (strcmp(tmp,"") == 0)
    {
        printf("ERROR:\t\tKEYWORD '%s' NOT DEFINED IN THE FITS HEADER\n",
               TIME_EXPO_KEYW);
        return FAILURE;
    }
    else
    {
        //Check the exposure time value
        if (sscanf(tmp, "%f", &exposureTime) != 1)
        {
            printf("ERROR:\t\tEXPOSURE TIME VALUE IS INCORRECT\n");
            return FAILURE;
        }
        else
        {
            if ((exposureTime*1000>20000)||(exposureTime*1000<20))
            {
                printf("ERROR:\t\tEXPOSITION TIME OUT OF RANGE\n"); 
                return FAILURE;
            }
        }
    }

    //check OBS_MODE 
    if (fitsFindInHeader(fitsHeader, OBS_MODE, &acquisitionMode[0]) == FAILURE)
    {
        return FAILURE;
    }

    if (strcmp(acquisitionMode,"") == 0)
    {
        printf("ERROR:\t\tKEYWORD '%s' NOT DEFINED IN THE FITS HEADER\n",
               OBS_MODE);
        return FAILURE;
    }

    //Check Camera Number
    if (fitsFindInHeader(fitsHeader, CAMERA_NUMBER,
                         &cameraNumberString[0]) == FAILURE)
    {
        return FAILURE;
    }

    if (strcmp(cameraNumberString,"") == 0)
    {
        printf("ERROR:\t\tKEYWORD '%s' NOT DEFINED IN THE FITS HEADER\n",
               CAMERA_NUMBER);
        return FAILURE;
    }
    else
    {
        if(sscanf(cameraNumberString,"%d",&cameraNumber) != 1)
        {
            printf("ERROR:\t\tKEYWORD '%s' UNKNOWN VALUE\n",CAMERA_NUMBER);
            return FAILURE;
        }
        else
        {
            if ((cameraNumber != MASTER_CAMERA_NUMBER) &&
                (cameraNumber != SLAVE_CAMERA_NUMBER))
            {
                printf("ERROR:\t\tKEYWORD '%s' UNKNOWN VALUE\n", CAMERA_NUMBER);
                return FAILURE;
            }
        }
    }
   
    free(mainFitsHeader);
    mainFitsHeader = calloc(strlen(fitsHeader),sizeof(char));
    if (mainFitsHeader == NULL)
    {
        printf("FATAL ERROR:\tCALLOC ERROR\n");
        exit(FAILURE);
    }

    strcpy(mainFitsHeader,fitsHeader);

    if (initAcquisition() == FAILURE)
    {
        return FAILURE;
    }
    else
    {
        return SUCCESS;
    }
}

int main()
{
    char    fitsHeader[FITS_HEADER_SIZE]; //FITS HEADER
    int     socketUCCI, sockWait;         //SOCKET DESCRIPTOR
    struct  sockaddr_in sin;              //BORDEL POUR SOCKET
    struct  sockaddr_in pin;
    int     addrLen;
    char    *ack;
    int     received = 0;
    int     i;
    int     foundEndMsg = FALSE;
    int     position = 0;

    //Allocate message memory
    ack = calloc(ACKNOWLEDGE_SIZE, sizeof(char));
    if (ack == NULL)
    {
        printf("FATAL ERROR:\tCALLOC ERROR\n");
        exit(FAILURE);
    }
            
    //Get a internet socket
    if ((socketUCCI = socket(AF_INET, SOCK_STREAM, 0)) == -1)
    {
        perror("FATAL ERROR:\tSOCKET UCCI");
        exit(FAILURE);
    }

    //Complete the socket structure
    bzero(&sin, sizeof(sin));
    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = INADDR_ANY;
    sin.sin_port = htons(PORT);

    //bind the socket
    if (bind(socketUCCI,(const struct sockaddr *)&sin, sizeof(sin)) == -1)
    {
        perror("FATAL ERROR:\tBIND SOCKETUCCI: ");
        close(socketUCCI);
        exit(FAILURE);
    }
    
    //Listen the socket
    if (listen(socketUCCI, 1) == -1)
    {
        perror("FATAL ERROR:\tLISTEN SOCKETUCCI: ");
        close(socketUCCI);
        exit(FAILURE);
    }

    while(1)
    {
        //Wait for a client
        if ((sockWait = accept(socketUCCI,(struct sockaddr *) &pin,
                               &addrLen)) == -1)
        {
            perror("FATAL ERROR:\tACCEPTING CONNECTION: ");
            close(socketUCCI);
            exit(FAILURE);
        }
        if (DEBUG_MODE == TRUE)
        printf("INFORMATION:\tConnection received\n");

        
        //Check if the transmission is ended
        if (DEBUG_MODE == TRUE)
        {
            printf("INFORMATION:\tWaiting for the end of the message\n");
        }


        //Check if the transmission is ended
        foundEndMsg = FALSE;
        while (foundEndMsg == FALSE)
        {
            //Initialize reception buffer
            memset (fitsHeader,0,FITS_HEADER_SIZE*sizeof(char)); 

            //Get the data reception acknowledge from the client
            if((received = recv(sockWait, fitsHeader, FITS_HEADER_SIZE,
                                MSG_PEEK)) == -1)
            {
                perror("FATAL ERROR:\tRECEIVING DATA: ");
                close(socketUCCI);
                close(sockWait);
                exit(FAILURE);
            }
            i=0;
            while((i < received) && (foundEndMsg == FALSE))
            {   
                if (fitsHeader[i] == END_MSG_CHAR)
                {
                    position = i;
                    foundEndMsg = TRUE;
                }
                i++;
            }
        }
        if (DEBUG_MODE == TRUE)
            printf("INFORMATION:\tEnd of the message\n");

        //Read the message
        if((recv(sockWait, fitsHeader, position+1,0)) == -1)
        {
            perror("FATAL ERROR:\tRECEIVING DATA: ");
            close(socketUCCI);
            close(sockWait);
            exit(FAILURE);
        }
        fitsHeader[position] = NULL;
        
        if (DEBUG_MODE == TRUE)
        printf("INFORMATION:\tReception of %d char \n", received);

        //Initialize reception buffer
        memset (ack,0,ACKNOWLEDGE_SIZE*sizeof(char)); 
        
        //Return an aknowledge of reception
        strcpy(ack, ACKNOLEDGE_RECEPTION);
        if (send(sockWait, ack, strlen(ACKNOLEDGE_RECEPTION), 0) == -1)
        {
            perror("FATAL ERROR:\tSENDING DATA: ");
            close(socketUCCI);
            close(sockWait);
            exit(FAILURE);
        }
        if (send(sockWait, END_MSG_STRING, 1, 0) == -1)
        {
            perror("FATAL ERROR:\tSENDING DATA: ");
            close(socketUCCI);
            close(sockWait);
            exit(FAILURE);
        }

        if (DEBUG_MODE == TRUE)
        printf("INFORMATION:\tFirst ucci acknowledge sent: '%s' \n", ack);

        //Initialize reception buffer
        memset(ack,0,ACKNOWLEDGE_SIZE*sizeof(char));
        
        //Process received data
        if (processData(fitsHeader) ==  FAILURE)
        {
            //Return an aknowledge of FAILURE
            strcpy(ack,ACKNOLEDGE_EXECUTION_ERR);
        }
        else
        {
            //Return an aknowledge of FAILURE
            strcpy(ack,ACKNOLEDGE_EXECUTION_OK);
        }

        //Send Acknowledge
        if (send(sockWait, ack, strlen(ack), 0) == -1)
        {
            perror("FATAL ERROR:\tSENDING DATA: ");
            close(socketUCCI);
            close(sockWait);
            exit(FAILURE);
        }
        if (send(sockWait, END_MSG_STRING, 1, 0) == -1)
        {
            perror("FATAL ERROR:\tSENDING DATA: ");
            close(socketUCCI);
            close(sockWait);
            exit(FAILURE);
        }

        if (DEBUG_MODE == TRUE)
            printf("INFORMATION:\tSecond ucci acknowledge sent: '%s' \n", ack);

        close(sockWait);
    }
    close(socketUCCI);
    return SUCCESS;
}
