
/**
 * Title:        Interface-Utilisateur<p>
 * Description:  Programmation d'une interface-utilisateur pour le telescope.<p>
 * Copyright:    Copyright (c) Miguel-Grassin-Minec-Debin<p>
 * Company:      Themis<p>
 * @author Miguel-Grassin-Minec-Debin
 * @version 1.0
 */

package iu;

import java.lang.Math;

/**
 * Permet de convertir des coordonnees entre differents systemes :
 * <ul>
 * <li>equatorial = (X (arcsec), Y (arcsec))<ul>
 *   <li>X=0 au centre, + vers l'est cleste
 *   <li>Y=0 au centre, + vers le nord cleste</ul>
 * <li>polar = heliocentric-radial coordinates = (Polarang (deg), SVector (arcsec))<ul>
 *   <li>SVector=0 au bord, + vers l'extrieur
 *   <li>Polarang=0 au nord solaire, + en sens antihoraire</ul>
 * <li>heliographic = heliographic coordinates = (Latitude (deg), Longitude (deg))<ul>
 *   <li>Latitude=0 au centre, + vers le nord solaire
 *   <li>Longitude=0 au centre, + vers l'ouest solaire</ul>
 * <li>heliocentric = heliocentric-cartesian coordinates = (Latitude (deg), Longitude (deg))<ul>
 *   <li>Latitude=0 au centre, + vers le nord cleste (?)
 *   <li>Longitude=0 au centre, + vers l'ouest cleste (?)</ul>
 * </ul>
 * @todo Attention : latitude et longitude inverses (X, Y) !
 */
public class Coordonnees {
  /***
   * premiere coordonnee
   * @see Coordonnees
   */
  public double X;
  /***
   * deuxieme coordonnee
   * @see Coordonnees
   */
  public double Y;

  private static double CB0=0.995, SB0=-.0967; // cos(B0), sin(B0)
  private static double P0=-9.5; // en deg
  private static double CP0=1, SP0=0; // cos(P0), sin(P0)
  private static double B0=0; // en deg 
  private static double R0=974.7; // en arcsec
  private static double LgCarr=254.04; // en deg

  private static java.text.DecimalFormat format = new java.text.DecimalFormat("####.####");
  private static final double DtoR = Math.PI/180.0;

  /**
   * Attention : avant d'effectuer une conversion, il faut initialiser les parametres
   * @see #setParameters(String)
   */
  public Coordonnees(double X, double Y) {
    this.X = X;
    this.Y = Y;
  }

  public String toString() {
    return "("+X+","+Y+")";
  }

  /**
   * Initialise les parametres solaires utilises pour les conversions.
   * Les parametres utilises sont : <ul>
   * <li> B0 = tilt of the solar north rotational axis toward the observer (deg)
   * <li> P0 = tilt of the solar north rotational axis toward the side (deg)
   * <li> R0 = rayon solaire apparent (arcsec)
   * <li> LgCarr = longitude du mridien central (deg)
   * </ul>
   * Doit etre appele avant d'utiliser les methodes de conversion.
   * @param DAO la chaine renvoyee telle quelle par l'UCCI
   */
  public static void setParameters(String DA0)
    throws IndexOutOfBoundsException, NumberFormatException {
    DA0 = DA0.substring(DA0.indexOf("DA3"));
    setP0(Double.parseDouble(DA0.substring(24, 31)));
    setB0(Double.parseDouble(DA0.substring(32, 39)));
    setLgCarr(Double.parseDouble(DA0.substring(40, 47)));
    setR0(Double.parseDouble(DA0.substring(54, 59)));
    AppIU.setStatus("Solar parameters updated.");
  }

  private static void setP0(double P) {
    P0 = P;
    CP0 = Math.cos(P*DtoR);
    SP0 = Math.sin(P*DtoR);
    System.out.println("solar_P0="+P);
  }

  public static float getP0() { return new Double(P0).floatValue(); }

  public static float getB0() { return new Double(B0).floatValue(); }

  private static void setB0(double B) {
    B0 = B;
    CB0 = Math.cos(B*DtoR);
    SB0 = Math.sin(B*DtoR);
    System.out.println("solar_B0="+B);
  }

  private static void setLgCarr(double L) {
    LgCarr = L;
    System.out.println("Lgcarr="+L);
  }

  private static void setR0(double R) {
    R0 = R;
    System.out.println("solar_R="+R);
  }

  /**
   * conversion equatorial -> polaire
   */
  public Coordonnees equatToPol() {
//System.out.println("[Coordonnees.equatToPol] X="+X+", Y="+Y);
    double rho = Math.sqrt(X*X + Y*Y);
    double theta = 0;
//System.out.println("[Coordonnees.equatToPol] rho="+rho);

    if (rho == 0d) theta = P0;
    else theta = Math.asin(X/rho)/DtoR;
    if (Y < 0) theta = 180. - theta;
    theta -= P0;
    if (theta < 0) theta += 360.;

    return new Coordonnees(theta, rho - R0);
  }

  /**
   * conversion equatorial -> heliographique
   * @exception ArithmeticException Si le point est en dehors du soleil, on ne
   * peut pas le coder en heliographique.
   */
  public Coordonnees equatToHeliog() throws ArithmeticException {
//System.out.println("[Coordonnees.equatToHeliog] X="+X+", Y="+Y);
    // equatorial -> heliocentric
    double x = -CP0*X + SP0*Y;
    double y =  SP0*X + CP0*Y;
//    double x = -CP0*X - SP0*Y;
//    double y = -SP0*X + CP0*Y;
    // heliocentric -> heliographic
    double z = Math.sqrt(R0*R0 - x*x - y*y);
    if (Double.isNaN(z)) throw new ArithmeticException("Out of bounds coordinate : z");
//System.out.println("[Coordonnees.equatToHeliog] x="+x+", y="+y+", z="+z);
    double theta = Math.asin((y*CB0 + z*SB0) / R0);
    if (Double.isNaN(theta)) throw new ArithmeticException("Out of bounds coordinate : theta");
    double alpha = z*CB0 - y*SB0;
    double phi;
    if (alpha == 0d) phi = Math.PI/2;
    else phi = Math.atan(x / alpha);
    // rad -> deg et apparent -> absolu (Carrington)
    theta /= DtoR;
    phi = phi /DtoR + LgCarr;
    if (phi>360) phi -= 360;
    if (phi<0) phi += 360; 
    return new Coordonnees(phi, theta);
  }

  /**
   * conversion polaire -> equatorial
   */
  public Coordonnees polToEquat() {
//System.out.println("[Coordonnees.polToEquat] X="+X+", Y="+Y);
    Y = Y+R0;
    return new Coordonnees(
      Y * Math.sin((X+P0)*DtoR),
      Y * Math.cos((X+P0)*DtoR)
    );
  }

  /**
   * conversion polaire -> heliographique
   * @exception ArithmeticException Si le point est en dehors du soleil, on ne
   * peut pas le coder en heliographique.
   */
  public Coordonnees polToHeliog() throws ArithmeticException {
//System.out.println("[Coordonnees.polToHeliog] X="+X+", Y="+Y);
    return polToEquat().equatToHeliog();
  }

  /**
   * conversion heliographique -> equatorial
   */
  public Coordonnees heliogToEquat() {
//System.out.println("[Coordonnees.heliogToEquat] X="+X+", Y="+Y);
    // absolu (Carrington) -> apparent
    X -= LgCarr;
    // deg -> rad
    X *= DtoR;
    Y *= DtoR;
    // heliographic -> heliocentric
    double x = R0 * Math.cos(Y) * Math.sin(X);
    double y = R0 * (Math.sin(Y)*CB0 - Math.cos(Y)*Math.cos(X)*SB0);
//System.out.println("[Coordonnees.heliogToEquat] x="+x+", y="+y);
    // heliocentric -> equatorial
    return new Coordonnees(
      -CP0*x + SP0*y,
      SP0*x + CP0*y
//      CP0*x + SP0*y,
//      SP0*x - CP0*y
    );
  }

  /**
   * conversion polar -> heliocentrique
   */
  public Coordonnees polToHelioc() {
System.out.println("[Coordonnees.polToHelioc] X="+X+", Y="+Y);
    return polToHeliog().heliogToHelioc();
  }

  /**
   * conversion equatorial -> heliocentrique
   */
  public Coordonnees equatToHelioc() {
//System.out.println("[Coordonnees.equatToHelioc] X="+X+", Y="+Y);
    return new Coordonnees(
      -CP0*X + SP0*Y,
       SP0*X + CP0*Y
    );
  }

  /**
   * conversion heliocentrique -> equatorial
   */
  public Coordonnees heliocToEquat() throws ArithmeticException {
//System.out.println("[Coordonnees.heliocToEquat] X="+X+", Y="+Y);
    return new Coordonnees(
      -CP0*X + SP0*Y,
      SP0*X + CP0*Y
    );
  }

  /**
   * conversion heliocentrique -> polar
   */
  public Coordonnees heliocToPol() {
//System.out.println("[Coordonnees.heliocTopol] X="+X+", Y="+Y);
    return heliocToHeliog().heliogToPol();
  }

  /**
   * conversion heliocentrique -> heliographique
   */
  public Coordonnees heliocToHeliog() throws ArithmeticException {
//System.out.println("[Coordonnees.heliocToHeliog] X="+X+", Y="+Y);
    double z = Math.sqrt(R0*R0 - X*X - Y*Y);
    if (Double.isNaN(z)) throw new ArithmeticException("Out of bounds coordinate : z");
    double theta = Math.asin((Y*CB0 + z*SB0) / R0);
    if (Double.isNaN(theta)) throw new ArithmeticException("Out of bounds coordinate : theta");
    double alpha = z*CB0 - Y*SB0;
    double phi;
    if (alpha == 0d) phi = Math.PI/2;
    else phi = Math.atan(X / alpha);
    // rad -> deg et apparent -> absolu (Carrington)
    theta /= DtoR;
    phi = phi /DtoR + LgCarr;
    if (phi>360) phi -= 360;
    if (phi<0) phi += 360; 
    return new Coordonnees(phi, theta);
  }

  /**
   * conversion heliographique -> heliocentrique
   */
  public Coordonnees heliogToHelioc() {
//System.out.println("[Coordonnees.heliogToHelioc] X="+X+", Y="+Y);
    // absolu (Carrington) -> apparent
    X -= LgCarr;
    // deg -> rad
    X *= DtoR;
    Y *= DtoR;
    // heliographic -> heliocentric
    return new Coordonnees(
      R0 * Math.cos(Y) * Math.sin(X),
      R0 * (Math.sin(Y)*CB0 - Math.cos(Y)*Math.cos(X)*SB0)
    );
  }

  /**
   * conversion heliographique -> polaire
   */
  public Coordonnees heliogToPol() {
//System.out.println("[Coordonnees.heliogToPol] X="+X+", Y="+Y);
    return heliogToEquat().equatToPol();
  }

/**
 * renvoie une chane reprsentant le double avec 2 chiffres aprs la virgule
 */
  public static String doubleFormat(double x) {
//System.out.println("[Coordonnees.doubleFormat] "+format.format(x));
    return format.format(x);
  }
}
