/********************************************************************;;#h#>
-------------
HEAD_OF_FILE: gen_entete.c
-------------

-Authors: C. Janneteau

-Purpose:

This file contains the functions used to read the value of the FITS keywords
in the tables (Tb_header of def_mcle.c) and write the FITS headers in
memory.

-Functions:

gen_header
Get_pos_kwd_in_List_ordo
Get_pos_kwd_tab_header
write_all_fen_header
write_header_line
write_header_simple_line
write_header_equal_line

-Comments:

Those functions can be used for both 'Top' and 'Dyn' headers. The variable
'Type_header h_type' is used to distinghish them.

-Keywords:

FITS, header.

;;#h#<*********************************************************************/

#include <stdio.h>
#include <strings.h>

#include <sun.h>
#include <acq.h>
#include <enrg.h>
#include <errno.h>
#include <the_errno.h>
#include <the_time.h>

extern  Erreur          the_errno;
extern  char            Liste_mcle[NB_MCLE][TAILLE_MCLE];
extern  Ind_mcle        Liste_ordo[NB_TYPE_HEADER][TAILLE_MAX_LIST_ORDO];
extern  Param_info      Tb_header_princ[NB_MCLE+NB_MCLE_DEPEND_FEN*NB_MAX_FEN];

extern FILE             *fd_spy;

/*****************************************************************;;#f#>
HEAD_OF_FUNCTION:  gen_header

-Purpose:

-Prototype:

int gen_header(Pt_header,Tb_header,Pt_enrg,Liste_ordo,h_type);

-Parameters:

Mem_header *Pt_header: pointeur sur le header en zone memoire
Param_info *Tb_header: tableau contenant la liste des kwd a mettre dans les headers
Tb_enrg    *Pt_enrg: pointeur vers info d'enregistrement
Ind_mcle   **Liste_ordo: tableau des mots cle exceptionnels (ordonnes)
Type_header h_type: type de l'entete a creer (dynamique ou top)

-Return:

ERROR en cas d'erreur ...sinon OK.

-Example:

->Pour creer une entete-memoire principale (top):
retour = gen_header(Pt_top_header,Tb_header_princ,Pt_enrg,Liste_ordo,TOP_HEADER);
->Pour creer une entete-memoire dynamique (dyn):
retour = gen_header(Pt_dyn_header,Tb_header_dyn,Pt_enrg,Liste_ordo,DYN_HEADER);

;;#f#<******************************************************************/
int gen_header(Pt_header,Tb_header,Pt_enrg,Liste_ordo,h_type)
Mem_header *Pt_header;
Param_info *Tb_header;
Tb_enrg    *Pt_enrg; 
Ind_mcle   Liste_ordo[NB_TYPE_HEADER][TAILLE_MAX_LIST_ORDO];
Type_header h_type;
{
  int ii,index , kk, retour;
  long nb_ligne, position[NB_MAX_FEN],nb_position,posi;
  Param_info mot_end;
  char message[TAILLE_MES];
  

  
  /*On determine le nombre de ligne a ecrire dans les entetes*/
  switch(h_type)
    {
    case TOP_HEADER:
      {
	nb_ligne = Pt_enrg->nb_li_princ;
	sprintf(message,"\n>>>> Dans gen_header: nb ligne top header %i",nb_ligne);
	message_spy(fd_spy,message);
	break;
      }
    case DYN_HEADER:
      {
	nb_ligne = Pt_enrg->nb_li_dyn;
	sprintf(message,"\n Dans gen_header : nombre ligne entete dyn: %i",nb_ligne);

	break;
      }
    default:
      {
	/*Type de header inconnu....!*/
	
	message_spy(fd_spy,"\ndans gen_header(): type de header inconnu...ni TOP_HEADER ni DYN_HEADER");
	the_errno.code=E_UNK_HEAD;
	the_errno.ctxt=FORMATAGE;
	return(ERROR);
	break;
      }
    }
  
  /*Initialisation des champ de_fen et du champ common*/
  strcpy(Pt_header->common,"\0");
  for(ii=0;ii<NB_MAX_FEN;ii++)
    {
      strcpy(Pt_header->de_fen[ii],"\0");
    }
  

  
  /*boucle sur le tableau contenant les infos pour le header principal*/
  
  /*
   * ETAPE 1
   *
   */

  /*Recherche des mots cles exceptionnels:...c'est a dire ceux qui doivent*/
  /*apparaitre a une position bien precise dans le header.*/
  /*Pour eux, la ligne est ecrite dans TOUS les*/
  /*champs 'de_fen' de la stucture Pt_header. Ceci permet de respecter */
  /*la contrainte de position dans le header.*/
 
  index = 0;
  while( (index<TAILLE_MAX_LIST_ORDO) && (Liste_ordo[h_type][index] != NOTH) )
    {

      
      /*Recherche du mot cle dans le tableau...*/
      retour = Get_pos_kwd_tab_header(Tb_header,nb_ligne,Liste_ordo[h_type][index],position,&nb_position);
      if(retour == ERROR)
	{
	  return(ERROR);
	}
      
      for(kk=0;kk<nb_position;kk++)
	{
	  
	  /*ecriture dans l'entete memoire*/
	  if(Tb_header[position[kk]].fenetre == -1)
	    {
	      /*le mot cle n'est pas dedie a une fenetre en particulier. Il est */
	      /*donc ecrit dans tous les cham de_fen...*/
	      
	      retour = write_all_fen_header(Tb_header[position[kk]],Pt_header, Pt_enrg);
	      if(retour == ERROR)
		{
		  return(ERROR);
		}
	    }
	  else
	    {
	      retour = write_header_line(Tb_header[position[kk]],Pt_header->de_fen[Tb_header[position[kk]].fenetre]);
	      if(retour == ERROR)
		{
		  return(ERROR);
		}
	    }
	}
      
      index++;
	
    }
  

 

  /*
   * ETAPE 2
   *
   */

  
  /*Maintenant que tous les mots cles exceptionnels sont ecrits dans les entetes*/
  /*memoires (champ: de_fen)...on peut ecrire les autres mots cles(sauf END)....*/
  


  for(ii=0;ii<nb_ligne;ii++)
    {
      /*read the information for the table and copy it into the header*/
      /*On test la valeur du mot cle pour eviter de retraiter les mots */
      /*cle exceptionnels..de plus on ne traitera END qu'a la fin...*/
      
      retour = Get_pos_kwd_in_List_ordo(Liste_ordo[h_type],Tb_header[ii].indice,&posi);
      if( retour == ERROR)
	{
	  return(ERROR);
	}
      
      if(( posi == -1) && (Tb_header[ii].indice != _END))
	{

	  if(Tb_header[ii].fenetre == -1)
	    {
	      /*le mot cle n'est pas dedie a une fenetre en particulier. Il est */
	      /*donc ecrit dans le champ 'common'...*/
	      
	      retour = write_header_line(Tb_header[ii],Pt_header->common);
	      if(retour == ERROR)
		{
		  return(ERROR);
		}
	    }
	  else
	    {
	      retour = write_header_line(Tb_header[ii],Pt_header->de_fen[Tb_header[ii].fenetre]);
	      if(retour == ERROR)
		{
		  return(ERROR);
		}
	    }
	}
      
    }
  
  /*
   * ETAPE 3
   *
   */
  
  /*Enfin...maintenant que tous les mots cle sont dans les entetes memoires il suffit*/
  /*d'ajouter le dernier: _END dans le champ common car ce dernier est ecrit apres le */
  /*champ de_fen dans le fichier. De plus la function complete_men_header_princ n'ecrit*/
  /*que dans les champs de_fen!*/
  

  
  	
  mot_end.indice = _END;
  mot_end.type = TYPE_CHFITS;
  strcpy(mot_end.value.type_chfits,"");
  mot_end.fenetre = -1; /*inutile...mais pour le respect des regles...*/ 
  
  retour = write_header_line(mot_end,Pt_header->common);
  if(retour == ERROR)
    {
      return(ERROR);
    }
  
  
  /*Sortie de la fonction..sans erreur...*/
  
  return(OK);
  
  
}

/*****************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: Get_pos_kwd_in_List_ordo
-----------------
-Purpose:

Get the position of a keyword in a list of the exceptional keywords.
It is a way to see if it is in the list or not...hahaha!

-Prototype:

int Get_pos_kwd_in_List_ordo(Liste,kwd,position)

-Parameters:

Ind_mcle *Liste:
Ind_mcle kwd:
long *position:

-Return:

OK.

;;#f#<*****************************************************************/
int Get_pos_kwd_in_List_ordo(Liste,kwd,position)
Ind_mcle *Liste;
Ind_mcle kwd;
long *position;
{
  long ii;
  

  
  ii=0;
  
  while((Liste[ii] != kwd) && (ii<TAILLE_MAX_LIST_ORDO))
    {
      ii++;
    }
  if(ii == TAILLE_MAX_LIST_ORDO)
    {
      /*kwd n'est pas dans Liste*/
      *position = -1;
    }
  else
    {
      /*kwd est dans Liste a la position ii*/
      *position = ii;
    }

  return(OK);
}


/*****************************************************************;;#f#>
HEAD_OF_FUNCTION: Get_pos_kwd_tab_header
;;#f#<*****************************************************************/
int Get_pos_kwd_tab_header(Tb_header,taille_tb_header,kwd,position,nb_position)
Param_info *Tb_header;
long taille_tb_header;
Ind_mcle   kwd;
long position[NB_MAX_FEN];
long *nb_position;
{
  int ii;
  

  
  *nb_position = 0;
  
  for(ii=0;ii<taille_tb_header;ii++)
    {
      if(Tb_header[ii].indice == kwd)
	{
	  position[*nb_position] = ii;
	  (*nb_position)++;
	}
    }

  return(OK);
}



/*****************************************************************;;#f#>
HEAD_OF_FUNCTION:  write_all_fen_header
;;#f#<******************************************************************/
int write_all_fen_header(Mot_cle,Pt_header,Pt_enrg)
Param_info Mot_cle;
Mem_header *Pt_header;
Tb_enrg    *Pt_enrg; 
{
  int num_fen, retour;
  

  
  for(num_fen=0;num_fen<NB_MAX_FEN;num_fen++)
    {
      if(Pt_enrg->numfen[num_fen] == 1)
	{
	  /*la fenetre numero 'num_fen' est a enregistrer...*/
	  retour = write_header_line(Mot_cle,Pt_header->de_fen[num_fen]);
	  if(retour == ERROR)
	    {
	      return(ERROR);
	    }
	}
    }
  
  return(OK);
}

/*****************************************************************;;#f#>
HEAD_OF_FUNCTION: write_header_line

-Purpose:
Ecrit une ligne au format fits dans un header FITS. Elle appelle
write_header_equal_line si la ligne est au format "KWD = ..." ou bien
write_header_simple_line si la ligne est au format "KWD ....".
;;#f#<******************************************************************/
int write_header_line(Mot_cle,header_block)
Param_info Mot_cle;
char *header_block;
{
  int retour;
  

  
  switch(Mot_cle.indice)
    {
    case _END:
      {

	/*Affecte la valeur des champs Mot_cle.type et Mot_cle.value avant*/
	/*d'appeler write_header_simple_line*/
	Mot_cle.type = TYPE_CHFITS; /*inutile...mais bon...*/
	strcpy(Mot_cle.value.type_chfits,"");
	retour = write_header_simple_line(Mot_cle,header_block);
	if(retour == ERROR)
	  {
	    return(ERROR);
	  }
	break;
      }
    default :
      {
	retour = write_header_equal_line(Mot_cle,header_block);
	if(retour == ERROR)
	  {
	    return(ERROR);
	  }
	break;
      }
    }
  
  
  return(OK);
}

/*****************************************************************;;#f#>
HEAD_OF_FUNCTION:  write_header_simple_line
-Purpose:
ecrit la ligne FITS simple: "KWD ...." (ex: END)
-Comment:
;;#f#<******************************************************************/
int write_header_simple_line(Mot_cle,header_block)
Param_info Mot_cle;
char *header_block;
{
  int ii;
  char ligne[TAILLE_LIGNE];
  char keyword[TAILLE_MCLE];


  
  /*stockage de mot cle dans une chaine de characteres*/
  strcpy(keyword,Liste_mcle[Mot_cle.indice]);

  message_spy(fd_spy,"\n>>>>>WHSIMPLELINE -> ");
  message_spy(fd_spy,keyword);
        
  /*Ecriture des informations importantes*/
  sprintf(ligne,"%-8.8s  %-70.70s",keyword,Mot_cle.value.type_chfits);
  
  /*Concatenation de la ligne au header...*/
  /*ATTENTION: strcat ne retourne pas de code d'erreur. Si ca plante ici*/
  /*verifier que la taille de header_bloc est suffisante ....!!!! */
  strcat(header_block,ligne);
	
  return(OK);
}




/*****************************************************************;;#f#>
HEAD_OF_FUNCTION:  write_header_equal_line
-Purpose:
ecrit les ligne fits au format "KWD = ......" 
;;#f#<******************************************************************/

int write_header_equal_line(Mot_cle,header_block)
Param_info Mot_cle;
char *header_block;
{
  int ii;
  char ligne[TAILLE_LIGNE];
  char buf_tmp[TAILLE_LIGNE];
  char keyword[TAILLE_MCLE];


  
  /*initialisation de la chaine utiliser pour competer la ligne avec des SP (0X20)*/
  for(ii=0;ii<TAILLE_LIGNE;ii++)
    {
      buf_tmp[ii] = 0X20;
    }

  /*stockage de mot cle dans une chaine de characteres*/
  strcpy(keyword,Liste_mcle[Mot_cle.indice]);

  message_spy(fd_spy,"\n>>>>>WHEQUALLINE -> ");
  message_spy(fd_spy,keyword);

  switch(Mot_cle.type)
    {
    case TYPE_CHAR:
      {
	/*Ecriture des informations importantes*/
	sprintf(ligne,"%-8.8s= %20c",keyword,Mot_cle.value.type_char);
	
	/*completion de la ligne avec des SP - OBLIGATOIRE*/
	strncat(ligne,buf_tmp,TAILLE_LIGNE-strlen(ligne));
	
	/*Concatenation de la ligne au header...*/
	/*ATTENTION: strcat ne retourne pas de code d'erreur. Si ca plante ici*/
	/*verifier que la taille de header_bloc est suffisante ....!!!! */
	strcat(header_block,ligne);
	
	break;
      }
    case TYPE_LONG:
      {
	/*Ecriture des informations importantes*/
	sprintf(ligne,"%-8.8s= %20d",keyword,Mot_cle.value.type_long);
	
	/*completion de la ligne avec des SP - OBLIGATOIRE*/
	strncat(ligne,buf_tmp,TAILLE_LIGNE-strlen(ligne));
	
	/*Concatenation de la ligne au header...*/
	/*ATTENTION: strcat ne retourne pas de code d'erreur. Si ca plante ici*/
	/*verifier que la taille de header_bloc est suffisante ....!!!! */
	strcat(header_block,ligne);
	
	break;
      }
    case TYPE_DOUBLE:
      {
	/*Ecriture des informations importantes*/
	sprintf(ligne,"%-8.8s= %20f",keyword,Mot_cle.value.type_double);
	
	/*completion de la ligne avec des SP - OBLIGATOIRE*/
	strncat(ligne,buf_tmp,TAILLE_LIGNE-strlen(ligne));
	
	
	/*Concatenation de la ligne au header...*/
	/*ATTENTION: strcat ne retourne pas de code d'erreur. Si ca plante ici*/
	/*verifier que la taille de header_bloc est suffisante ....!!!! */
	strcat(header_block,ligne);
	
	break;
      }
    case TYPE_CHFITS:
      {
	/*Ecriture des informations importantes*/
	sprintf(ligne,"%-8.8s= %-70.70s",keyword,Mot_cle.value.type_chfits);
		
	/*Concatenation de la ligne au header...*/
	/*ATTENTION: strcat ne retourne pas de code d'erreur. Si ca plante ici*/
	/*verifier que la taille de header_bloc est suffisante ....!!!! */
	strcat(header_block,ligne);
	break;
      }
    default :
      {
	the_errno.code=E_UNK_KEYTYPE;
	the_errno.ctxt=FORMATAGE;
	  
	return(ERROR);
	break;
      }
    }
  
  return(OK);
}


/*****************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: complete_mem_header_princ()
-----------------

-Purpose:

Complete (in memory) the top header (Pt_top_header) in order to add the
information that can just be obtained after the output fils have been
created (FILENAME,FILEORIG,DATE,DATE_END).

-Prototype:

-Parameters:

-Return:
Ok ...or ERROR if there is a problem!
-Comments:

!!! This function must write only the de_fen fields (and not the common)
of the top memory header. this all kwd_info.fenetre fields have to be
different from -1 here. this in order to avoid to write info after the END!!!

Of course, the normal way to use this function is to launch it just after
the gen_header() and open_file() functions ...and before writting the header
in the file(function: write_fits_header_in_file()).
why?...because Pt_top_header is allocated in gen_header and that the useful
information for this function are computed thanks to open_file();
;;#f#<*****************************************************************/
int complete_mem_header_princ(Pt_header,Pt_enrg)
Mem_header *Pt_header;
Tb_enrg *Pt_enrg;
{
  int ii, retour;
  chfits filename, filedate;
  Param_info kwd_info;
  

  
  /*pour chaque fenetre enregistree, recuperer le nom du fichier*/
  /*associe et par la meme sa date de creation...pour les inserer*/
  /*dans l'entete principale*/
  
  for(ii=0;ii<NB_MAX_FEN;ii++)
    {
      if(Pt_enrg->numfen[ii]==1)
	{
	  /*catch the name of the file*/
	  
	  strcpy(filename,"\0");
	  sprintf(filename,"'%s'",Pt_enrg->nomfich[ii]);
	  
	  /*write FILENAME in the (memory) top header*/
	  
	  kwd_info.indice = _FILENAME;
	  kwd_info.type = TYPE_CHFITS;
	  strcpy(kwd_info.value.type_chfits,filename);
	  kwd_info.fenetre = ii;
	  
	  retour = write_header_line(kwd_info,Pt_header->de_fen[kwd_info.fenetre]);
	  if(retour == ERROR)
	    {
	      return(ERROR);
	    }
	  


	  /*write FILEORIG in the (memory) top header*/
	  
	  kwd_info.indice = _FILEORIG;
	  kwd_info.type = TYPE_CHFITS;
	  strcpy(kwd_info.value.type_chfits,filename);
	  kwd_info.fenetre = ii;
	  
	  retour = write_header_line(kwd_info,Pt_header->de_fen[kwd_info.fenetre]);
	  if(retour == ERROR)
	    {
	      return(ERROR);
	    }
	  

	  
	  /*catch the DATE from the filename*/
	  
	  strcpy(filedate,"\0");
	  sprintf(filedate,"'%s'",Pt_enrg->datefich[ii]);
	  
	  
	  /*write DATE in the (memory) top header*/
	  
	  kwd_info.indice = _DATE;
	  kwd_info.type = TYPE_CHFITS;
	  strcpy(kwd_info.value.type_chfits,filedate);
	  kwd_info.fenetre = ii;
	  
	  retour = write_header_line(kwd_info,Pt_header->de_fen[kwd_info.fenetre]);
	  if(retour == ERROR)
	    {
	      return(ERROR);
	    }
	  

	  
	  /*write DATE_END in the (memory) top header*/
	  
	  kwd_info.indice = _DATE_END;
	  kwd_info.type = TYPE_CHFITS;
	  strcpy(kwd_info.value.type_chfits,filedate);
	  kwd_info.fenetre = ii;
	  
	  retour = write_header_line(kwd_info,Pt_header->de_fen[kwd_info.fenetre]);
	  if(retour == ERROR)
	    {
	      return(ERROR);
	    }
	  

	  
	}
    }
  
  return(OK);
}
/**********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: clean_mem_header_princ_before_complete()
-----------------


-Purpose:
Effacer les 4 dernieres lignes des champs "de_fen" de "Pt_top_header" correspondant
aux parametres FILENAME, FILEORIG, DATE et DATE_END. Ceci afin que ces champs puissent
etre recrit (via complete_mem_header_princ())apres chaque ouverture d'un nouveau
fichier,sans avoir besoin de reconstruire toutela structure Pt_top_header.
CECI EST UTILE EN PARTICULIER POUR Le CAS THEMISFF=1, ou un nouveau fichier est cree pour
chaque acquisition.

*********************************************************************************;;#f#<*/
int clean_mem_header_princ_before_complete(Pt_header,Pt_enrg)
Mem_header *Pt_header;
Tb_enrg *Pt_enrg;
{
 
  int ii;
  char* ptr_deb_ligne = NULL;
  char* ptr_fin_ligne = NULL;

  /*boucle sur toutes les fenetres enregistrees*/
  for(ii=0;ii<NB_MAX_FEN;ii++)
    {
      if(Pt_enrg->numfen[ii]==1)
	{
	  /*eliminer la ligne FILENAME si elle existe dans Pt_header->de_fen[ii]*/
	  ptr_deb_ligne = NULL;
	  ptr_fin_ligne = NULL;
	  ptr_deb_ligne = strstr(Pt_header->de_fen[ii],Liste_mcle[_FILENAME]);
	  if(ptr_deb_ligne != NULL)
	    {
	      /*FILENAME a ete trouve....donc enlever cette ligne*/
	      ptr_fin_ligne = ptr_deb_ligne + TAILLE_LIGNE;
	      strcpy(ptr_deb_ligne,"\0");
	      strcat(Pt_header->de_fen[ii],ptr_fin_ligne);
	    }
	  
	  /*eliminer la ligne FILEORIG si elle existe dans Pt_header->de_fen[ii]*/
	  ptr_deb_ligne = NULL;
	  ptr_fin_ligne = NULL;
	  ptr_deb_ligne = strstr(Pt_header->de_fen[ii],Liste_mcle[_FILEORIG]);
	  if(ptr_deb_ligne != NULL)
	    {
	      /*FILEORIG a ete trouve....donc enlever cette ligne*/
	      ptr_fin_ligne = ptr_deb_ligne + TAILLE_LIGNE;
	      strcpy(ptr_deb_ligne,"\0");
	      strcat(Pt_header->de_fen[ii],ptr_fin_ligne);
	    }
	  
	  /*eliminer la ligne DATE si elle existe dans Pt_header->de_fen[ii]*/
	  ptr_deb_ligne = NULL;
	  ptr_fin_ligne = NULL;
	  ptr_deb_ligne = strstr(Pt_header->de_fen[ii],Liste_mcle[_DATE]);
	  if(ptr_deb_ligne != NULL)
	    {
	      /*DATE a ete trouve....donc enlever cette ligne*/
	      ptr_fin_ligne = ptr_deb_ligne + TAILLE_LIGNE;
	      strcpy(ptr_deb_ligne,"\0");
	      strcat(Pt_header->de_fen[ii],ptr_fin_ligne);
	    }
	  
	  /*eliminer la ligne DATE_END si elle existe dans Pt_header->de_fen[ii]*/
	  ptr_deb_ligne = NULL;
	  ptr_fin_ligne = NULL;
	  ptr_deb_ligne = strstr(Pt_header->de_fen[ii],Liste_mcle[_DATE_END]);
	  if(ptr_deb_ligne != NULL)
	    {
	      /*_DATE_END a ete trouve....donc enlever cette ligne*/
	      ptr_fin_ligne = ptr_deb_ligne + TAILLE_LIGNE;
	      strcpy(ptr_deb_ligne,"\0");
	      strcat(Pt_header->de_fen[ii],ptr_fin_ligne);
	    }
	}
    }

  return(OK);

}



/**********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: alloc_Mem_header()
-----------------

*********************************************************************************;;#f#<*/
Mem_header* alloc_Mem_header(h_type)
Type_header h_type;
{
  Mem_header *Pt_header;
  int ii;
  


  switch(h_type)
    {
    case TOP_HEADER:
      {

	
	/*allocation memoire de l'entete principale*/
	Pt_header=(Mem_header *)malloc(sizeof(Mem_header));
	if(Pt_header == NULL)
	  {
	    the_errno.code=errno;
	    the_errno.ctxt=FORMATAGE;
	    return(NULL);
	  }
	/*choix de la taille du champ common*/
	Pt_header->common_size = (3*TAILLE_BLOC_FITS)+1;
	if((Pt_header->common = (char *) malloc((Pt_header->common_size)*sizeof(char))) == NULL)
	  {
	    free(Pt_header);
	    the_errno.code = errno;
	    return(NULL);
	  }	
        
	break;
      }
    case DYN_HEADER:
      {


	/*allocation memoire de l'entete dynamique (sous-entete)*/
	Pt_header=(Mem_header *)malloc(sizeof(Mem_header));
	if(Pt_header == NULL)
	  {
	    the_errno.code=errno;
	    the_errno.ctxt=FORMATAGE;
	    return(NULL);
	  }
        /*choix de la taille du champ common*/
        Pt_header->common_size = TAILLE_BLOC_FITS+1;
        if((Pt_header->common = (char *) malloc((Pt_header->common_size)*sizeof(char))) == NULL)
          {
	    free(Pt_header);
            the_errno.code = errno;
            return(NULL);
          }
	
        break;
      }
    default:
      {
        /*Type de header inconnu....!*/
	
        message_spy(fd_spy,"\ndans alloc_Mem_header(): type de header inconnu...ni TOP_HEADER ni DYN_HEADER");

        return(NULL);
        break;
      }
    }
  
  return(Pt_header);
  
}
/**********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION:free_Mem_header
-----------------
*********************************************************************************;;#f#<*/
int free_Mem_header(Pt_header)
Mem_header *Pt_header;
{

  
  if(Pt_header != NULL)
    {
      if(Pt_header->common != NULL)
	{
	  free(Pt_header->common);
	}
      free(Pt_header);
    }
  
  return(OK);
}

/**********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: build_Tb_header_dyn
-----------------

-Purpose:

This function build/rebuild the table Tb_header_dyn which is then used to build the dynamical
headers in memory.

-Prototype:
-Parameters:
-Return:

Ok..or ERROR if there is a problem.

-Comments:



This function has to be used each time a new acquisition is done. Once the VME has updated
the structures corresponding to dynamical information, this function is used to create
Tb_dyn_header. Then the memory header is created and finally written in the files just
before the data.

;;#f#<**********************************************************************************/
int build_Tb_header_dyn(Tb_header_dyn,Pt_acq,Pt_enrg,Pt_ctrl,header)
Param_info *Tb_header_dyn;
Tb_acq     *Pt_acq;
Tb_enrg    *Pt_enrg;
Tb_ctrl    *Pt_ctrl;
int         header;
{
char message[TAILLE_MES];  

  
  /*initialisation: Repositionner l'indice d'ecriture derriere les infos*/
  /*fixes et semi-statiques presentes dans le tableau des entetes dynamiques i.e*/
  /*(XTENSION, BITPIX, NAXIS, NAXIS1, NAXIS2, NAXIS3 etc...)*/
  Pt_enrg->nb_li_dyn = Pt_enrg->indstart_dyn; 
  
  sprintf(message,"\nbuild_Tb_header_dyn: init de nblig dyn: %i", Pt_enrg->nb_li_dyn);
  message_spy(fd_spy,message);
  /*Ajout du Mot cle DATE_OBS dans le tableau ....*/

  if(header==DYN_HEADER)
    {
      if(ERROR == store_date_obs(Tb_header_dyn,Pt_acq,Pt_enrg))
	  return(ERROR);
    }
  
  /*Ajout des informations correspondant a TOUS les index */
  /*Note: on n'utilise pas les flags DFLAG_XXXXX!*/
   if(ERROR == store_index(Tb_header_dyn,Pt_acq,Pt_enrg,header))
    {
      return(ERROR);
    }
   if(ERROR == store_step(Tb_header_dyn,Pt_acq,Pt_enrg,header))
     {
      return(ERROR);
     }
   if(ERROR == store_stepX(Tb_header_dyn,Pt_acq,Pt_enrg,header))
     {
      return(ERROR);
     }
   if(ERROR == store_stepY(Tb_header_dyn,Pt_acq,Pt_enrg,header))
     {
      return(ERROR);
     }
   /*
     Ajout du mot cle LAMBD pour l'IO qui doit envoyer l'indice
     de la fenetre pour creer finalement le mot cle INLAMBD dans
     header FITS...

     if(ERROR == store_lambda(Tb_header_dyn,Pt_acq,Pt_enrg,header))
     {
     return(ERROR);
     }
     Retrait du mot cle STOKES du formatage 27/4/2000

     if(ERROR == store_ana(Tb_header_dyn,Pt_acq,Pt_enrg,header))
     {
     return(ERROR);
     }
   
     Si nouveaux disponibles: ajout des parametres poursuites ppou
     if((Pt_ctrl->dyn_flags[DFLAG_PPOU] == DYN_FLAG_ON)||(Pt_acq->themisff == FITS1))
   */
   
   /*de nouvelles infos sont disponibles...les prendre*/
   if(ERROR == store_ppou(Tb_header_dyn,Pt_acq,Pt_enrg,header))
     {
       return(ERROR);
     }
   /*on remet alors le flag OFF*/
   Pt_ctrl->dyn_flags[DFLAG_PPOU] = DYN_FLAG_OFF;
   
   /*
    retrait de POSQWP1 et 2 le 27/4/2000
    remis le 1 fev 2001
    
   Si nouveaux disponibles: ajout des parametres poursuites plam
   */

   if(Pt_ctrl->dyn_flags[DFLAG_PLAM] == DYN_FLAG_ON)
   {
     /*de nouvelles infos sont disponibles...les prendre*/

   if(ERROR == store_plam(Tb_header_dyn,Pt_acq,Pt_enrg,header))
   {
   return(ERROR);
   }
   /*on remet alors le flag OFF*/
   Pt_ctrl->dyn_flags[DFLAG_PLAM] = DYN_FLAG_OFF;
   }
   

   sprintf(message,"\nbuild_Tb_header_dyn: sortie nblig dyn: %i", Pt_enrg->nb_li_dyn);
   message_spy(fd_spy,message);
   return(OK);
}

/********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: store_ppou
-----------------

-Purpose:

This function reads the parameters, corresponding the the flag DFLAG_PPOU, in the data 
stuctures and adds them in the table for the dynamical headers (Tb_header_dyn).

;;#f#<********************************************************************************/
int store_ppou(Tb_header_dyn,Pt_acq,Pt_enrg,header)
Param_info *Tb_header_dyn;
Tb_acq  *Pt_acq;
Tb_enrg *Pt_enrg;
int header;
{
 
  long ligne;

  if(header==TOP_HEADER)
    ligne=Pt_enrg->nb_li_princ;
  else
    ligne=Pt_enrg->nb_li_dyn;
  
  /*_PREFENTE*/
  Tb_header_dyn[ligne].indice=_PREFENTE;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).prefente;
  ligne++;
  
  /*_POSROT*/
  Tb_header_dyn[ligne].indice=_POSROT;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).posrot;
  ligne++;
  
  /*_ELEVA*/
  Tb_header_dyn[ligne].indice=_ELEVA;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).eleva;
  ligne++;
      
  /*_AZIMUT*/
  Tb_header_dyn[ligne].indice=_AZIMUT;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).azimuth;
  ligne++;
  
  /*_ALPHA*/
  Tb_header_dyn[ligne].indice=_ALPHA;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).alpha;
  ligne++;

  /*_DELTA*/
  Tb_header_dyn[ligne].indice=_DELTA;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).delta;
  ligne++;

  /*_POLARANG*/
  Tb_header_dyn[ligne].indice=_POLARANG;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).polarang;
  ligne++;

  /*_SVECTOR*/
  Tb_header_dyn[ligne].indice=_SVECTOR;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).svector;
  ligne++;

  /*_LONGITUD*/
  Tb_header_dyn[ligne].indice=_LONGITUD;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).longitud;
  ligne++;  

  /*_LATITUD*/
  Tb_header_dyn[ligne].indice=_LATITUD;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).latitud;
  ligne++;

  /*DIST_EW */
  Tb_header_dyn[ligne].indice=_DIST_EW;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).dist_ew;
  ligne++;

  /*DIST_NS */
  Tb_header_dyn[ligne].indice=_DIST_NS;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).dist_ns;
  ligne++;

  /*LONGCARR */
  Tb_header_dyn[ligne].indice=_LONGCARR;
  Tb_header_dyn[ligne].type=TYPE_DOUBLE;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_double=(Pt_acq->dyn).longcarr;
  ligne++;

  
  if(header==TOP_HEADER)
    Pt_enrg->nb_li_princ=ligne;
  else
    Pt_enrg->nb_li_dyn=ligne;
  

  
  return(OK);
}

/********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: store_da0
-----------------

-Purpose:

This function reads the parameters in the data 
stuctures and adds them in the table for the dynamical headers (Tb_header_dyn).

;;#f#<********************************************************************************/
int store_da0(Tb_header,Pt_acq,Pt_enrg,header)
Param_info *Tb_header;
Tb_acq  *Pt_acq;
Tb_enrg *Pt_enrg;
int header;
{
 
  long ligne;

  if(header==TOP_HEADER)
    ligne=Pt_enrg->nb_li_princ;
  else
    ligne=Pt_enrg->nb_li_dyn;
  
  /*_SOLAR_P0*/
  Tb_header[ligne].indice=_SOLAR_P0;
  Tb_header[ligne].type=TYPE_DOUBLE;
  Tb_header[ligne].fenetre=NOTH;
  Tb_header[ligne].value.type_double=(Pt_acq->dyn).solar_p0;
  ligne++;
  
  /*_SOLAR_B0*/
  Tb_header[ligne].indice=_SOLAR_B0;
  Tb_header[ligne].type=TYPE_DOUBLE;
  Tb_header[ligne].fenetre=NOTH;
  Tb_header[ligne].value.type_double=(Pt_acq->dyn).solar_b0;
  ligne++;
  
  /*_SOLROT_N*/
  Tb_header[ligne].indice=_SOLROT_N;
  Tb_header[ligne].type=TYPE_LONG;
  Tb_header[ligne].fenetre=NOTH;
  Tb_header[ligne].value.type_long=(Pt_acq->dyn).solar_n;
  ligne++;
      
  /*_SOLAR_R*/
  Tb_header[ligne].indice=_SOLAR_R;
  Tb_header[ligne].type=TYPE_DOUBLE;
  Tb_header[ligne].fenetre=NOTH;
  Tb_header[ligne].value.type_double=(Pt_acq->dyn).solar_r;
  ligne++;
  
  if(header==TOP_HEADER)
    Pt_enrg->nb_li_princ=ligne;
  else
    Pt_enrg->nb_li_dyn=ligne;
  
  return(OK);
}

/********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: store_plam
-----------------

-Purpose:

This function reads the parameters, corresponding to the flag DFLAG_PLAM, in the data 
stuctures and adds them in the table for the dynamical headers (Tb_header_dyn).

;;#f#<********************************************************************************/
int store_plam(Tb_header_dyn,Pt_acq,Pt_enrg,header)
Param_info *Tb_header_dyn;
Tb_acq  *Pt_acq;
Tb_enrg *Pt_enrg;
int header;
{
  /* fonction non lance car POSQWP1 et 2 enleve */
  long ligne;

  if(header==TOP_HEADER)
    ligne=Pt_enrg->nb_li_princ;
  else
    ligne=Pt_enrg->nb_li_dyn;

  
  
  Tb_header_dyn[ligne].indice=_POSQWP1;
  Tb_header_dyn[ligne].type=TYPE_LONG;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_long=(Pt_acq->dyn).posqwp1;
  ligne++;

     
  Tb_header_dyn[ligne].indice=_POSQWP2;
  Tb_header_dyn[ligne].type=TYPE_LONG;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_long=(Pt_acq->dyn).posqwp2;
  ligne++;

    
  if(header==TOP_HEADER)
    Pt_enrg->nb_li_princ=ligne;
  else
    Pt_enrg->nb_li_dyn=ligne;
  
  
  
  return(OK);
  
}

/********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: store_date_obs
-----------------

-Purpose:
Compute and store DATE-OBS in the  table for the dynamical headers (Tb_header_dyn).
;;#f#<********************************************************************************/
int store_date_obs(Tb_header_dyn,Pt_acq,Pt_enrg)
Param_info *Tb_header_dyn;
Tb_acq  *Pt_acq;
Tb_enrg *Pt_enrg;
{
  struct timeval time2; 
  int retour;
  time_t cl1,cl2;
  struct tm *filetime;
  int year,month,day,hr,mn,sc,msc;
  chfits dateobs;
  

  
  /*initialisation */
  the_errno.code=0;
  if((filetime=(struct tm*)malloc(sizeof(struct tm))) == NULL)
    {
      the_errno.code=errno;
      the_errno.ctxt=FORMATAGE;
      return(ERROR);
    }
  
  /* recuperation du temps */
    retour=gettimeofday(&time2,NULL);
    if (retour == ERROR)
    {
      the_errno.code=errno;
      the_errno.ctxt=FORMATAGE;
      free(filetime);
      return(ERROR);
    }
  
    /* mise en forme */ 
  cl2=time2.tv_sec;
  filetime = gmtime(&cl2);
  year=filetime->tm_year+SIECLE;
  month=filetime->tm_mon+1;
  day=filetime->tm_mday;
  hr=filetime->tm_hour;
  mn=filetime->tm_min;
  sc=filetime->tm_sec;
  msc=time2.tv_usec/10000;
  
  /*creation de la chaine contenant l'heure au bon format*/
  sprintf(dateobs,"\0");
  sprintf(dateobs,"'%i-%.2i-%.2iT%.2i:%.2i:%.2i.%.2i'\0",year,month,day,hr,mn,sc,msc);

  /*Ecriture dans le tableau des entetes dynamiques*/
  Tb_header_dyn[Pt_enrg->nb_li_dyn].indice=_DATE_OBS;
  Tb_header_dyn[Pt_enrg->nb_li_dyn].type=TYPE_CHFITS;
  Tb_header_dyn[Pt_enrg->nb_li_dyn].fenetre=NOTH;
  strcpy(Tb_header_dyn[Pt_enrg->nb_li_dyn].value.type_chfits,dateobs);
  Pt_enrg->nb_li_dyn++;

      
  free(filetime);
  
  return(OK);

}

/********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: store_index
-----------------

-Purpose:
This function reads the parameters, corresponding to the flags DFLAG_INDEX, in the data 
stuctures and adds them in the table for the dynamical headers (Tb_header_dyn).

;;#f#<********************************************************************************/
int store_index(Tb_header_dyn,Pt_acq,Pt_enrg,header)
Param_info *Tb_header_dyn;
Tb_acq  *Pt_acq;
Tb_enrg *Pt_enrg;
int header;
{
  long ligne;

  if(header==TOP_HEADER)
    ligne=Pt_enrg->nb_li_princ;
  else
    ligne=Pt_enrg->nb_li_dyn;

  
  /* _SED_IND */
  Tb_header_dyn[ligne].indice=_SEQ_IND;
  Tb_header_dyn[ligne].type=TYPE_LONG;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_long = (Pt_acq->index).seq_ind;
  ligne++;

  
  /* _INDREG */
  Tb_header_dyn[ligne].indice=_INDREG;
  Tb_header_dyn[ligne].type=TYPE_LONG;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_long = (Pt_acq->index).ind_reg;
  ligne++;


  if(header==TOP_HEADER)
    Pt_enrg->nb_li_princ=ligne;
  else
    Pt_enrg->nb_li_dyn=ligne;
  
  
  return(OK);
}

/********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: store_step
-----------------

-Purpose:
This function reads the parameters, corresponding to the flags DFLAG_STEP, in the data 
stuctures and adds them in the table for the dynamical headers (Tb_header_dyn).

;;#f#<********************************************************************************/
int store_step(Tb_header_dyn,Pt_acq,Pt_enrg,header)
Param_info *Tb_header_dyn;
Tb_acq  *Pt_acq;
Tb_enrg *Pt_enrg;
int header;
{
  long ligne;

  if(header==TOP_HEADER)
    ligne=Pt_enrg->nb_li_princ;
  else
    ligne=Pt_enrg->nb_li_dyn;

  
  /*_INSTEP*/
  Tb_header_dyn[ligne].indice=_INDSTEP;
  Tb_header_dyn[ligne].type=TYPE_LONG;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_long = (Pt_acq->index).indstep;
  ligne++;


  if(header==TOP_HEADER)
    Pt_enrg->nb_li_princ=ligne;
  else
    Pt_enrg->nb_li_dyn=ligne;
  
  
  return(OK);
}

/********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: store_stepX
-----------------

;;#f#<********************************************************************************/
int store_stepX(Tb_header_dyn,Pt_acq,Pt_enrg,header)
Param_info *Tb_header_dyn;
Tb_acq  *Pt_acq;
Tb_enrg *Pt_enrg;
int header;
{
  long ligne;

  if(header==TOP_HEADER)
    ligne=Pt_enrg->nb_li_princ;
  else
    ligne=Pt_enrg->nb_li_dyn;

  
  /*_INSTEPX*/
  Tb_header_dyn[ligne].indice=_INDSTEPX;
  Tb_header_dyn[ligne].type=TYPE_LONG;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_long = (Pt_acq->index).indstepx;
  ligne++;


  if(header==TOP_HEADER)
    Pt_enrg->nb_li_princ=ligne;
  else
    Pt_enrg->nb_li_dyn=ligne;
  
  return(OK);
}

/********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: store_stepY
-----------------

;;#f#<********************************************************************************/
int store_stepY(Tb_header_dyn,Pt_acq,Pt_enrg,header)
Param_info *Tb_header_dyn;
Tb_acq  *Pt_acq;
Tb_enrg *Pt_enrg;
int header;
{
  long ligne;

  if(header==TOP_HEADER)
    ligne=Pt_enrg->nb_li_princ;
  else
    ligne=Pt_enrg->nb_li_dyn;

  
  /*_INSTEPY*/
  Tb_header_dyn[ligne].indice=_INDSTEPY;
  Tb_header_dyn[ligne].type=TYPE_LONG;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_long = (Pt_acq->index).indstepy;
  ligne++;


  if(header==TOP_HEADER)
    Pt_enrg->nb_li_princ=ligne;
  else
    Pt_enrg->nb_li_dyn=ligne;
  
  return(OK);
}

/********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: store_lambda
-----------------

-Purpose:
This function reads the parameters, corresponding to the flags DFLAG_INDLAMBD, in the
data stuctures and adds them in the table for the dynamical headers (Tb_header_dyn).

;;#f#<********************************************************************************/
int store_lambda(Tb_header_dyn,Pt_acq,Pt_enrg,header)
Param_info *Tb_header_dyn;
Tb_acq  *Pt_acq;
Tb_enrg *Pt_enrg;
int header;
{

  long ligne;

  if(header==TOP_HEADER)
    ligne=Pt_enrg->nb_li_princ;
  else
    ligne=Pt_enrg->nb_li_dyn;

  
  /*_INDLAMBD*/
  Tb_header_dyn[ligne].indice=_INDLAMBD;
  Tb_header_dyn[ligne].type=TYPE_LONG;
  Tb_header_dyn[ligne].fenetre=NOTH;
  Tb_header_dyn[ligne].value.type_long = (Pt_acq->index).indlambd;
  ligne++;


  if(header==TOP_HEADER)
    Pt_enrg->nb_li_princ=ligne;
  else
    Pt_enrg->nb_li_dyn=ligne;
  

  
  return(OK);
}

/********************************************************************************;;#f#>
-----------------
HEAD_OF_FUNCTION: store_ana
-----------------

-Purpose:
This function reads the parameters, corresponding to the flags DFLAG_STOKES, in the
data stuctures and adds them in the table for the dynamical headers (Tb_header_dyn).

;;#f#<********************************************************************************/
int store_ana(Tb_header,Pt_acq,Pt_enrg,header)
Param_info *Tb_header;
Tb_acq  *Pt_acq;
Tb_enrg *Pt_enrg;
int header;
{

  long ligne;
  char message[TAILLE_MES];

  if(header==TOP_HEADER)
      ligne=Pt_enrg->nb_li_princ;
  else
    ligne=Pt_enrg->nb_li_dyn;

  /*_STOKES*/
  Tb_header[ligne].indice=_STOKES;
  Tb_header[ligne].type=TYPE_LONG;
  Tb_header[ligne].fenetre=NOTH;
  Tb_header[ligne].value.type_long = (Pt_acq->index).stokes;
  ligne++;

  if(header==TOP_HEADER)
    Pt_enrg->nb_li_princ=ligne;
  else
    Pt_enrg->nb_li_dyn=ligne;
  

  return(OK);
}

