
static char *SccsId="@(#)Sun-inst-pont.c	1.12 04/27/00";

/*********************************************************************;;#h#>
-----------------
HEAD_OF_FILE: Sun-inst-pont.c
-----------------
-Authors:	Christian Miguel
-Library: Sun-inst-pont.a
-Purpose:
This file contains all the functions for The Ptsbs915 bus driver. Thanks to
those functions a program linking the library Sun-inst-pont.a can access the
memory of the VME machine, which must be shared (readable and writable )in the
acquisition program.

	
-Variables:
	
int		desc_pont	(dev of driver)
void 		*Pt_vme_base    (points on the first elt mapped)

Tb_point	*Pt_point;	(pointers on acq info) 
Tb_acq		*Pt_acq;		
Tb_ctrl_cam	*Pt_cam;
Tb_ctrl		*Pt_ctrl;
Datum		*Pt_data;

	 	
-Functions:

open_driver_pont
close_driver_pont
add_point
sub_point
map_mem
unmap_mem
map_mem_fils
installe_pont		  
ferme_pont		 
	 
-Comments:
Installing the mapping is quite tricky without those functions. It is very
important to follow the instructions to use those functions because they are
very low level. The first error may reboot the station.

There are two ways of installing the bus to the VME machine from the SUN:

1)
If only one process needs to map the memory, we are in the simpliest case.
The call of installe_pont installs everything.
It is possible to read and write in the shared memory using the pointers
on the acquisition info (if they are declared as extern in the other files).
WARNING: It is essential to call ferme_pont before exiting to close the
device.

2)
If 2 processes, at least, must access the shared memory, it is rather more
complicated.
One process, the father, must proceed as if it was the only one (like 1).
The other ones must use the function map_mem_fils instead of install_pont
and close the device using unmap_mem.

-Keywords:
Memory, VME, PTSBS915,Driver.
;;#h#<*********************************************************************/


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <fcntl.h>
#include <sys/mman.h>		/* pour les cstes utiles a mmap  */
#include <unistd.h>
#include <errno.h>              

#include	<sun.h>        /* not system section */
#include	<Sun-inst-pont.h>
#include 	<acq.h>
#include	<the_errno.h>



void 		*Pt_vme_base=NULL;  

Tb_point	*Pt_point=NULL;
Tb_acq		*Pt_acq=NULL;
Tb_ctrl_cam	*Pt_cam=NULL;
Tb_ctrl		*Pt_ctrl=NULL;
Datum		*Pt_data=NULL;

int 		desc_pont;


extern Erreur the_errno;    /* see errors library */


/*******************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION: open_driver_pont
------------------
-Purpose:
Open the driver of the "pont".

-Prototype:
int open_driver_pont()

-Return:
The return of the open function in the name of the driver DRIVER_PONT

;;#f#<*********************************************************************/


int open_driver_pont()
{
	return(open(DRIVER_PONT,O_RDWR));

}





/******************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION: close_driver_pont
------------------

-Purpose:
Close the driver of the "pont".

-Prototype:
int close_driver_pont()

-UsesVariables:
desc_pont:  The "file handle" of the driver.
-Return:
the return of the close function.
-Comments:
Note that this function takes no parameters : "close(desc_pont)" is 
directly used in the code.

;;#f#<******************************************************************/

int close_driver_pont()
{
	return(close(desc_pont));
}




/******************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION: add_point
------------------

-Purpose:	
Adding two pointers !
-Prototype:
	void * add_point(void *base,void *offset)
-Parameters:
	base:	one pointer
	offset:	the other one !

-Return:
The sum.
-Comments:
There is no error checking.

;;#f#<*********************************************************************/


void *add_point(base,offset)
void *base;
void *offset;
{
	return((void *)((u_long)offset+(u_long)base));
}


/*******************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION: sub_point
------------------

-Purpose:
Substracting pointers.
-Prototype:
	void * sub_point(void *base,void *offset)
-Parameters:
	base:	One pointer
	offset:	The one to substract.

-Return: 
The result.
-Comments:
There is no error checking

;;#f#<*******************************************************************/


void *sub_point(base,offset)
void *base;
void *offset;
{
	return((void *)((u_long)base-(u_long)offset));
}



 
/*******************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION: map_mem
------------------

-Purpose:
Mapping address, the lowest level function in this file.

-Prototype:
	int map_mem(u_long offset, int desc)

-Parameters:
offset: The physical address of the memory of the VME machine. It depends on
type of bus, 16 32 bits etc...
desc: is the real device.

-UsesVariables:

Pt_vme_base 	virtual @ of the beginning of the global data area
Pt_point	virtual @ of the table of the pointers
Pt_acq		virtual @ of the acquisition table
Pt_ctrl		virtual @ of the control table
Pt_data		virtual @ of the biginning of the data area
Pt_cam		virtual @ of the cameras control table

-UsesFunctions:
map_registres (int_reg.c)
x
-Return:
OK or ERROR. There is an error control (the_errno variable)


-Comments:
Not to be called directly. This functions not only maps the memory but maps the registers of the card to programm them.
Note that the standard function mmap(...) is used here.

;;#f#<*******************************************************************/

int map_mem(offset,desc) 
u_long offset; 
int desc; 
{ 
  caddr_t Registres;
  int i;

  if(ERROR == map_registres())
    return(ERROR);

  Pt_vme_base=mmap(0,TAILLE_MEM_COM,PROT_READ|PROT_WRITE,MAP_SHARED,desc,OFFSET_PORT_VME); 
       
  if((int)Pt_vme_base == MAP_FAILED) 
    { printf("\n ERROR mapmemoire"); 
      the_errno.code=errno; return(ERROR); 
    }

  Pt_point=add_point(Pt_vme_base,(void *)offset); 
  Pt_acq=(Tb_acq *)add_point(Pt_point->pt_tab_acq,Pt_vme_base); 
  Pt_ctrl=(Tb_ctrl *)add_point(Pt_point->pt_ctrl,Pt_vme_base);
  Pt_data=(Datum *)add_point(Pt_point->pt_data,Pt_vme_base);
  Pt_cam=(Tb_ctrl_cam *)add_point(Pt_point->pt_cam,Pt_vme_base);
 
  return(OK);
}

/*******************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION: unmap_mem
------------------

-Purpose:
End the mapping of memory.

-Prototype:
	int unmap_mem()

-UsesVariables:
 Pt_vme_base : virtual @ of the biginning of the global data area

-Return:
Ok or ERROR. There is an error control in this function.
-Comments:
Note that the standard function munmap(...) is used here.
;;#f#<*********************************************************************/


int unmap_mem()
{
	return(munmap(Pt_vme_base,TAILLE_MEM_COM));

}

/******************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION: map_mem_fils
------------------

-Purpose:

It is similar to the map_mem function but is to be used in a multi process
environment. It has to be used by the (or a) son process when both father and son(s) have to access the shared mmemory of the VME.

-Prototype:
	int map_mem_fils(ex_base,offset,desc)
-Parameters:
exbase: is the Pt_vme_base value obtained in the first map (calling of install_pont offset is the value of the Pt_point in the VME (VME address)).
desc:  is the device.

-UsesVariables:	
Pt_vme_base
Pt_point
Pt_acq
Pt_ctrl
Pt_data
Pt_cam

-UsesFunctions:
add_point
sub_point

-Return:
OK or ERROR there is an error control

-Comments:
Note that the standard function mmap(...) is used here.

;;#f#<*********************************************************************/


int map_mem_fils(ex_base,offset,desc)
u_long ex_base;
u_long offset;
int desc;
{ 
  u_short *Registres;
  int i;

  /* Pt_vme_base=Map_Memory(TAILLE_MEM_COM,desc,OFFSET_PORT_VME);*/
  Pt_vme_base=mmap(0,TAILLE_MEM_COM,PROT_READ|PROT_WRITE,MAP_SHARED,desc,OFFSET_PORT_VME);


  if((int)Pt_vme_base == ERROR)
    return(ERROR); 

  Pt_point=add_point(Pt_vme_base,(void *)offset);
  
  Pt_acq=(Tb_acq *)add_point(Pt_point->pt_tab_acq,
			     sub_point(Pt_vme_base,(void *)ex_base));
  Pt_ctrl=(Tb_ctrl *)add_point(Pt_point->pt_ctrl,
			       sub_point(Pt_vme_base,(void *)ex_base));
  Pt_data=(Datum *)add_point(Pt_point->pt_data,
			     sub_point(Pt_vme_base,(void *)ex_base));
  Pt_cam=(Tb_ctrl_cam *)add_point(Pt_point->pt_cam,
				  sub_point(Pt_vme_base,(void *)ex_base));
  return(OK);
}





/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION: installe_pont
------------------

-Purpose:
This function do every thing necessary to install the shared memory.

-Prototype:
	Tb_point *installe_pont(offset)
-Parameters:
offset: is the value of Pt_point in the VME (VME address value).

-UsesVariables:
desc_pont 
Pt_acq
Pt_ctrl
Pt_data
Pt_cam
Pt_Point
-UsesFunctions:
open_driver_pont
map_mem

-Return:
Pt_point, Sun address of Tb_point structure.

-Comments:
Once this function is called, the SUN can access the shared memory of the VME.

;;#f#<*********************************************************************/

Tb_point *installe_pont(offset)
u_long offset;
{ 
	desc_pont=open_driver_pont();

	if(ERROR==desc_pont)
	{
		printf("\n Erreur ouverture");
		the_errno.code=errno;
		return(NULL);
	}
	
	if(map_mem(offset,desc_pont)==ERROR)
	{
		printf("\n Erreur map");
		return(NULL);
	}
	
/* On met les adresses SUN dans la table des pointeurs */

	Pt_point->pt_tab_acq=Pt_acq;
	Pt_point->pt_ctrl=Pt_ctrl;
	Pt_point->pt_data=Pt_data;
	Pt_point->pt_cam=Pt_cam;

	return(Pt_point);
}





/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION: ferme_pont
------------------

-Purpose:
Ends the mapping after a call of Install_pont.

-Prototype:
	int ferme_pont()
-UsesVariables:
desc_pont
Pt_acq
Pt_ctrl
Pt_data
Pt_cam
Pt_vme_base
-UsesFunctions:
sub_point
-Comments:
It is really,really important to call this function after Install_pont
before exiting !!
Note that the standard function munmap is used here.
;;#f#<*********************************************************************/

int ferme_pont()
{	

	
/* on remet les adresses VME dans la table des pointeurs */

	Pt_point->pt_tab_acq=sub_point(Pt_acq,Pt_vme_base);
	Pt_point->pt_ctrl=sub_point(Pt_ctrl,Pt_vme_base);
	Pt_point->pt_data=sub_point(Pt_data,Pt_vme_base);
	Pt_point->pt_cam=sub_point(Pt_cam,Pt_vme_base);		
	 
	if(ERROR==munmap(Pt_vme_base,TAILLE_MEM_COM))
	{
		the_errno.code=errno;
		return(ERROR);
	}
 		
 	
	close(desc_pont);
	return(OK);
		
}





