
static char *SccsId="%W% %G%";

/********************************************************************;;#h#>
-----------------
HEAD_OF_FILE:action.c
-----------------
-Authors:      C Miguel
-Library:
             param.a
-Purpose:
This File contains the functions dealing with the keywords, each keyword
corresponds to an action for the program which reads the parameter file
(FILE_RAO_PAR).
The acquisition parameters must be read and stored in memory. Once a keyword
is found in the file, the program starts the corresponding action which is
named action_<keyword found>. All the following functions are stored in a
Variable the documentation is in the def_mcle.c file.


-Functions:
		action_modob
		action_naxis3
		action_themisff
		action_rien
		action_dingt
		action_obtur
		action_tpcam
		action_nocam
		action_nbcam
		action_nblig
		action_nbcol
		action_ligdb
		action_coldb
		action_sensl
		action_enreg
		action_visuf
		action_ctrac
		action_cadac
		action_nbfen
		action_nbacq
		action_end
	 
-Comments:

1)The functions returns:
	->	0 if ok 
	->	-1 if error (syntaxe ou smantique)
	->	+1 when the end keyword is found.
All the lines in the parameters file start with a meaningless letter (usually
"_"), this due to a difficult compatibility between the different languages
found in the data acquisition.


2)The acquisition parameter file (FILE_RAO_PAR) contains a keyword by
line. One can distinguish two types of line depending of the corresponding
keyword. 

->The first type as the format: <LISTING><H4>KEY  =      value</H4></LISTING>
->The second:<LISTING><H4>KEY indice =      value</H4></LISTIN

So 'action' functions which correspond to keyword of the second type will
call 'extrait_indice' (cf util_param.c) inn order to extract the indice.

3)Note that most of the 'action' functions defined here will use these three
constantes defined in param.h:

->TAILLE_VALU: Size of the string where will be stored the value read from
the file FILE_RAO_ACQ. This string will then be converted into integer.

->TAILLE_VAL: Size of the string (corresponding to the field value) read in
the file FILE_RAO_ACQ. Of course TAILLE_VALU >= TAILLE_VAL is preferable.

->DEBUT_VAL: starting position (in a line of the FILE_RAO_PAR file) from
which the field 'value' can be read. DEBUT_VAL should be high enough to 
cat only 'value' (and not the previous field 'key' and 'indice').  

-Keywords:
          keyword,parameter.
  	
;;#h#<*********************************************************************/



#include <stdio.h>
#include <stdlib.h>
#include <sys/ddi.h>

 
#include	<sun.h>
#include 	<acq.h>
#include 	<enrg.h>
#include 	<visu.h>
#include 	<param.h>
#include 	<camera.h>
#include 	<the_errno.h>

extern Erreur the_errno;
extern FILE *fd_spy;
extern Param_info Tb_header_princ[NB_MCLE+NB_MCLE_DEPEND_FEN*NB_MAX_FEN];
extern Param_info Tb_header_dyn [NB_LI_BLOC_FITS+NB_LI_BLOC_FITS*NB_MAX_FEN];


/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_modob
------------------

-Purpose:
        Sets the acquisition mode:
	
	MODE_RAO tunning, value 0.
	MODE_SOL full disk, value 1.
	MODE_STK stokesmetre, value 2.
	MODE_TRT real time reduction, value 3 (not implemented).
	MODE_REGL_SE full disk tunning, value 4.
	MODE_OBSCU dark current, value 5 (not implemented).

-Prototype:
int action_modob(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne:
Tb_enrg	*Pt_enrg:
Tb_visu	*Pt_visu:
Tb_acq	*Pt_acq:
Tb_ctrl_cam *Pt_cam:
-Example:
<LISTING><H4>
_MODOB   =                  0
</H4></LISTING>
;;#f#<*********************************************************************/

int action_modob(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char cval[TAILLE_VALU];
  char message[TAILLE_MES];
  long value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
 
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }
    
	
  if(Pt_acq!=NULL)
    {
      if( (value==MODE_RAO)||(value==MODE_STK))
	{
	  if(Pt_acq->modob==NOTH)
	    Pt_acq->modob=value;
	  else
	    {
	      the_errno.code=E_DECODE_REDUN;
	      return(ERROR);
	    }
	}
      else
	{
	  sprintf(message,"\n modob valeur: %i",value);

	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }

  if(Pt_visu!=NULL)
    Pt_visu->modob=value;


  return(OK);
	
}


action_naxis3(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{

  char 	cval[TAILLE_VALU];
  int 	ind=extrait_indice(ligne);
  long 	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  Tb_header_princ[Pt_enrg->nb_li_princ].indice=_NAXIS3;
  Tb_header_princ[Pt_enrg->nb_li_princ].type=TYPE_LONG;
  Tb_header_princ[Pt_enrg->nb_li_princ].fenetre=NOTH;
  Tb_header_princ[Pt_enrg->nb_li_princ].value.type_long=value;
  Pt_enrg->nb_li_princ++;
  
  Tb_header_dyn[Pt_enrg->nb_li_dyn].indice=_NAXIS3;
  Tb_header_dyn[Pt_enrg->nb_li_dyn].type=TYPE_LONG;
  Tb_header_dyn[Pt_enrg->nb_li_dyn].fenetre=NOTH;
  Tb_header_dyn[Pt_enrg->nb_li_dyn].value.type_long=value;
  (Pt_enrg->tab_phy).naxis3=value;
  Pt_enrg->nb_li_dyn++;
  Pt_enrg->indstart_dyn++;
  
  return(OK); 
}


action_themisff(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char cval[TAILLE_VALU];
  char message[TAILLE_MES];

  long value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
	
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }


  if(Pt_acq!=NULL)
    {
      if( (value==FITS0)||(value==FITS1)||(value==FITS2)||(value==FITS3))
	{
	  if(Pt_acq->themisff==NOTH)
	    {
	      Pt_acq->themisff=value;
	      Tb_header_princ[Pt_enrg->nb_li_princ].indice=_THEMISFF;
	      Tb_header_princ[Pt_enrg->nb_li_princ].type=TYPE_LONG;
	      Tb_header_princ[Pt_enrg->nb_li_princ].fenetre=NOTH;
	      Tb_header_princ[Pt_enrg->nb_li_princ].value.type_long=value;
	      Pt_enrg->nb_li_princ++;
	    }
	  else
	    {
	      the_errno.code=E_DECODE_REDUN;
	      return(ERROR);
	    }
	}
      else
	{
	  printf("\n value = %i",value);
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }

  if(Pt_visu!=NULL)
    Pt_visu->modob=value;


  return(OK);
}


/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_rien
------------------

-Purpose:
 Does obsolutely nothing, it is the default action.
-Prototype:
int action_rien(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Return:
Just returns OK !
;;#f#<*********************************************************************/

int action_rien(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
	return(OK);	 
}


/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_dingt
------------------

-Purpose:
       Sets the integration duration for each obturator (ms).
-Prototype:
int action_dintg(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne:
Tb_enrg	*Pt_enrg:
Tb_visu	*Pt_visu:
Tb_acq	*Pt_acq:
-Example:
The duration is set for the obturator 1 to 350 ms
<LISTING><H4>
_DINTG  1=                  350
</H4></LISTING>
;;#f#<*********************************************************************/

int action_dintg(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  int 	ind=extrait_indice(ligne);
  long  value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  if(Pt_acq!=NULL)
    {
      if( (ind>=0)&&(ind<NB_MAX_OBT) )
	{
	  Pt_acq->dintg[ind]=value;
	}
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  return(OK);
}
/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_obtur
------------------

-Purpose:
     Sets the number of obturator for each buffer (camera memory)
-Prototype:
int action_obtur(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Comments:
     The value 7 (obturator) keeps it open (life saving)
-Example:
 
     The buffer number 1 corresponds to the obturator 2
<LISTING><H4>
_OBTUR  1=                    2
</H4></LISTING>
;;#f#<*********************************************************************/

int action_obtur(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;
{
  char 	cval[TAILLE_VALU];
  int	ind=extrait_indice(ligne);
  long	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }
	
  if(Pt_cam!=NULL)
    {
      if(((ind>=0)&&(ind<NB_MAX_CAM))&&(value>=0)&&(value<NB_MAX_OBT) )
	{
	  Pt_cam->tab_reg_cam[ind].n_obt=value;
	}
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  return(OK);
}	
/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_tpcam
------------------

-Purpose:
 Sets the temperature for the cooling of the cameras
-Prototype:
int action_tpcam(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Example:

Temperature for camera 1
<LISTING><H4>
_TPCAM  1=                  350
</H4></LISTING>
;;#f#<*********************************************************************/

int action_tpcam(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;
{
  char 	cval[TAILLE_VALU];
  int	ind=extrait_indice(ligne);
  long	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  if(Pt_cam!=NULL)
    {
      if(((ind>=0)&&(ind<NB_MAX_CAM))&&(value<=TEMP_MAX)&&(value>=TEMP_MIN))
	{
	  Pt_cam->tab_reg_cam[ind].theta=value;
	}
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  return(OK);
}	

/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_nocam
------------------

-Purpose:
 Sets the camera number for a window of data
-Prototype:
int action_nocam(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :

-Example:

Window 1 is set to buffer 2
<LISTING><H4>
_NOCAM  1=                    2
</LISTING></H4>
;;#f#<*********************************************************************/


int action_nocam(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  int 	ind=extrait_indice(ligne);
  long 	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  int    i;
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  i=Pt_enrg->nb_li_princ;

  if((value>=0)&&(value<NB_MAX_CAM))
    {

      Pt_acq->tab_info_fen[ind].nocam=value;

      Tb_header_princ[i].indice=_NOCAM;
      Tb_header_princ[i].type=TYPE_LONG;
      Tb_header_princ[i].fenetre=ind;
      Tb_header_princ[i++].value.type_long=value;
      Pt_enrg->nb_li_princ=i;
      return(OK); 
    }
  else
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }
}

int action_lambd(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;


{
  char 	cval[TAILLE_VALU];
  int 	ind=extrait_indice(ligne);
  long 	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  int    i;
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    } 

  i=Pt_enrg->nb_li_princ;

  if((value>=0)&&(value<NB_MAX_CAM))
    {

      /*      Pt_acq->tab_info_fen[ind].indlambd=value;*/

      Tb_header_princ[i].indice=_INDLAMBD;
      Tb_header_princ[i].type=TYPE_LONG;
      Tb_header_princ[i].fenetre=ind;
      Tb_header_princ[i++].value.type_long=value;
      Pt_enrg->nb_li_princ=i;
      return(OK); 
    }
  else
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }
}





/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_nobuf
------------------

-Purpose:
 Sets the buffer number for a window of data
-Prototype:
int action_nobuf(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :

-Example:

Window 1 is set to buffer 2
<LISTING><H4>
_NOBUF  1=                    2
</LISTING></H4>
;;#f#<*********************************************************************/

int action_nobuf(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  int 	ind=extrait_indice(ligne);
  long 	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  int    i,j;
  char message[TAILLE_MES];
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  i=Pt_enrg->nb_li_princ;

  if(Pt_acq!=NULL)
    {
      if( (ind>=0)&&(ind<NB_MAX_FEN)&&(value>=0)&&(value<NB_MAX_CAM))
	{
	  Pt_acq->tab_info_fen[ind].nobuf=value;
	  Pt_acq->tab_info_fen[ind].noplan=PLAN_IMAGE;
	  
	  Tb_header_princ[i++].indice=_NOBUF;
	  Tb_header_princ[i++].indice=_NSHUTTER;
	  Tb_header_princ[i++].indice=_TEMPCAM;
	  Tb_header_princ[i++].indice=_EXPTIME;	  
	  Tb_header_princ[i++].indice=_NOWIN;
	 



	  
	  for(i=Pt_enrg->nb_li_princ;i<Pt_enrg->nb_li_princ+4;i++)
	    {
	      Tb_header_princ[i].type=TYPE_LONG;
	      Tb_header_princ[i].fenetre=ind;
	      Tb_header_princ[i].value.type_long=value;
	    }
	  
	  Tb_header_princ[i].type = TYPE_LONG;
	  Tb_header_princ[i].fenetre = ind;
	  Tb_header_princ[i].value.type_long = ind;
	  i++;
	  Pt_enrg->nb_li_princ=i;

	}
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  if(Pt_visu!=NULL)
    Pt_visu->numcam_de_donnee[ind]=value;
  return(OK); 
}

/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_nbcam
------------------

-Purpose:
 	Sets the number of cameras in use in the observation, the value must
	not exceed 32 (NB_MAX_CAM)
-Prototype:
int action_nbcam(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Example:
Use two cameras :
<LISTING> <H4>
_NBCAM   =                    2
</H4></LISTING>
;;#f#<*********************************************************************/

int action_nbcam(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  long 	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VAL));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  if(Pt_acq!=NULL)
    {
      if( (value>0)&&(value<NB_MAX_CAM) )
	{
	  Pt_acq->nbcam=value;

	  Tb_header_princ[Pt_enrg->nb_li_princ].indice=_NBCAM;
	  Tb_header_princ[Pt_enrg->nb_li_princ].type=TYPE_LONG;
	  Tb_header_princ[Pt_enrg->nb_li_princ].fenetre=NOTH;
	  Tb_header_princ[Pt_enrg->nb_li_princ].value.type_long=value;
	  Pt_enrg->nb_li_princ++;
	}
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  return(OK); 
}


/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_nblig
------------------

-Purpose:
Sets the number of lines of a window, the value must not exceed 512.
-Prototype:
int action_nblig(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Example:
The window 1 has 200 lines
<LISTING><H4>
_NBLIG  1=                  200
</H4></LISTING>
;;#f#<*********************************************************************/

int action_nblig(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  int 	ind=extrait_indice(ligne);
  long	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }
	
  if(Pt_acq!=NULL)
    {
      if( (ind>=0)&&(ind<NB_MAX_FEN)&&(value>0)&&(value<MAX_LIG_CCD) )
	{
	  Pt_acq->tab_fen[ind].nblig=value;
	
	  Tb_header_princ[Pt_enrg->nb_li_princ].indice=_NAXIS2;
	  Tb_header_princ[Pt_enrg->nb_li_princ].type=TYPE_LONG;
	  Tb_header_princ[Pt_enrg->nb_li_princ].fenetre=ind;
	  Tb_header_princ[Pt_enrg->nb_li_princ].value.type_long=value;
	  Pt_enrg->nb_li_princ++;
	  Tb_header_dyn[Pt_enrg->nb_li_dyn].indice=_NAXIS2;
	  Tb_header_dyn[Pt_enrg->nb_li_dyn].type=TYPE_LONG;
	  Tb_header_dyn[Pt_enrg->nb_li_dyn].fenetre=ind;
	  Tb_header_dyn[Pt_enrg->nb_li_dyn].value.type_long=value;
	  Pt_enrg->nb_li_dyn++;
	  Pt_enrg->indstart_dyn++;
	}
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }

  if(Pt_visu!=NULL)
    Pt_visu->tab_fen[ind].nblig=value;

  return(OK);
}

/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_nbcol
------------------

-Purpose:
Sets the number of columns in a window, it must not exceed 512.

-Prototype:
int action_nbcol(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Example:
Set window 1 with 380 colums.
<LISTING><H4>
_NBCOL  1=                  380
</H4></LISTING>

;;#f#<*********************************************************************/

int action_nbcol(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  int 	ind=extrait_indice(ligne);
  long	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  if(Pt_acq!=NULL)
    {
      if( (ind>=0)&&(ind<NB_MAX_FEN)&&(value>0)&&(value<=MAX_COL_CCD) )
	{
	  Pt_acq->tab_fen[ind].nbcol=value;

	  Tb_header_princ[Pt_enrg->nb_li_princ].indice=_NAXIS1;
	  Tb_header_princ[Pt_enrg->nb_li_princ].type=TYPE_LONG;
	  Tb_header_princ[Pt_enrg->nb_li_princ].fenetre=ind;
	  Tb_header_princ[Pt_enrg->nb_li_princ].value.type_long=value;
	  Pt_enrg->nb_li_princ++;
	  Tb_header_dyn[Pt_enrg->nb_li_dyn].indice=_NAXIS1;
	  Tb_header_dyn[Pt_enrg->nb_li_dyn].type=TYPE_LONG;
	  Tb_header_dyn[Pt_enrg->nb_li_dyn].fenetre=ind;
	  Tb_header_dyn[Pt_enrg->nb_li_dyn].value.type_long=value;
	  Pt_enrg->nb_li_dyn++;
	  Pt_enrg->indstart_dyn++;
	}
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  if(Pt_visu != NULL)
    Pt_visu->tab_fen[ind].nbcol=value;


  return(OK); 
}


/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_ligdb
------------------

-Purpose:
Sets the number of the first line of the window
-Prototype:
int action_ligdb(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Example:
Set the first line of window 1 to the line number 1.
<LISTING><H4>
_LIGDB  1=                    1
</H4></LISTING>

;;#f#<*********************************************************************/

int action_ligdb(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  int 	ind=extrait_indice(ligne);
  long	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  if(Pt_acq!=NULL)
    {
      if( (ind>=0)&&(ind<NB_MAX_FEN)&&(value>=0)&&(value<MAX_LIG_CCD) )
	{
	  Pt_acq->tab_fen[ind].ligdb=value;

	  Tb_header_princ[Pt_enrg->nb_li_princ].indice=_ROWSTART;
	  Tb_header_princ[Pt_enrg->nb_li_princ].type=TYPE_LONG;
	  Tb_header_princ[Pt_enrg->nb_li_princ].fenetre=ind;
	  Tb_header_princ[Pt_enrg->nb_li_princ].value.type_long=value;
	  Pt_enrg->nb_li_princ++;
	}
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  if(Pt_visu!=NULL)
    Pt_visu->tab_fen[ind].ligdb=value;

  return(OK);
}


/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_coldb
------------------

-Purpose:
Sets the number of the fist column of the window
-Prototype:
int action_coldb(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Example:
Set to 200 the first column of the window 1.
<LISTING><H4>
_COLDB  1=                  200
</H4></LISTING>

;;#f#<*********************************************************************/

int action_coldb(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  int 	ind=extrait_indice(ligne);
  long	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  if(Pt_acq!=NULL)
    {
      if( (ind>=0)&&(ind<NB_MAX_FEN)&&(value>=0)&&(value<MAX_COL_CCD) )
	{
	  Pt_acq->tab_fen[ind].coldb=value;

	  Tb_header_princ[Pt_enrg->nb_li_princ].indice=_COLSTART;
	  Tb_header_princ[Pt_enrg->nb_li_princ].type=TYPE_LONG;
	  Tb_header_princ[Pt_enrg->nb_li_princ].fenetre=ind;
	  Tb_header_princ[Pt_enrg->nb_li_princ].value.type_long=value;
	  Pt_enrg->nb_li_princ++;
	}
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  if(Pt_visu != NULL)
    Pt_visu->tab_fen[ind].coldb=value;

  return(OK);
}


/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_sensl
------------------

-Purpose:
Sets the way of reading the data in the buffers of the cameras and of cours
the way of writing them in the VME memory (shared memory).

0 means column by column (default value)
1 means line by line
-Prototype:
int action_sensl(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :

-Example:
Window 1 will be read column by column
<LISTING><H4>
_SENSL  1=                    0
</H4></LISTING>
;;#f#<*********************************************************************/

int action_sensl(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  int 	ind=extrait_indice(ligne);
  long 	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  if(Pt_acq!=NULL)
    {
      if( (value==SENS_LIG)||(value==SENS_COL) )
	Pt_acq->tab_info_fen[ind].sens=value;
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  return(OK); 
}


/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_enreg
------------------

-Purpose:
Means the window is to be recorded on file
-Prototype:
int action_enreg(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Example:
Window 2 is to be recorded
<LISTING><H4>
_ENREG  2=                    1
</H4></LISTING>
;;#f#<*********************************************************************/

int action_enreg(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  int 	ind=extrait_indice(ligne);
  long	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  char  message[TAILLE_MES];
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  if(Pt_enrg!=NULL)
    {
      if((ind>=0)&&(ind<NB_MAX_FEN)&&((value==1)||(value==0)) )
	{
	  if(Pt_enrg->numfen[ind]==NOTH)
	    {	   
	      Pt_enrg->numfen[ind]=value;
	      if(value==1)
		Pt_enrg->nbfen++;
	    }
	  else
	    {
	      the_errno.code=E_DECODE_REDUN;
	      return(ERROR);
	    }
	}
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}		
    }
  return(OK); 
}


/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_visuf
------------------

-Purpose:
Indicates which window is to be displaid on the screen.
Not in use at the moment
-Prototype:
int action_visuf(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :

;;#f#<*********************************************************************/

int action_visuf(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  int 	ind=extrait_indice(ligne);
  long	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  if(Pt_visu!=NULL)
    {
      if((ind>=0)&&(ind<NB_MAX_FEN)&&((value==1)||(value==0))&& 
	 (Pt_visu->nbfen<NB_MAX_FEN_VISU) )
	{
	  if(Pt_visu->numfen[ind]==NOTH) 
	    {

	      Pt_visu->numfen[ind]=value;
	      if(value==1)
		Pt_visu->nbfen++;
	    }
	  else
	    {
	      the_errno.code=E_DECODE_REDUN;
	      return(ERROR);
	    }
	}
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  return(OK); 
}



/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_ctrac
------------------

-Purpose:

How the data acquisition are started.

 ->CONTINU (i.e. 4) : means the acquisition never stops (the vme starts as 
soon as one is finished)
 ->CADENCEUR (i.e. 2):allows to performed a certain number of acquisition 
(defined by the keyword _NBACQ) specifying the speed of the acquisition thanks
to the keyword _CADAC.ands then stop.
 ->0 :is the simple mode, only one acquisition is made, when required.

-Prototype:
int action_ctrac(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Example:
The acquisition mode is the simple one
<LISTING><H4>
_CTRAC   =                    0
</H4></LISTING>

;;#f#<*********************************************************************/

int action_ctrac(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  long 	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  if(Pt_acq!=NULL)
    {
      if( (value==SIMPLE)||(value==CADENCEUR)||(value==CONTINU) )
	Pt_acq->ctrac=value;
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  return(OK); 
}


/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_cadac
------------------

-Purpose:
Sets the speed of the acquisition, the value is a duration in ms. This lap 
time must take into account the numeration (250ms) and the integration
time.

-Prototype:
int action_cadac(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Example:
One acquistion is made every 500 ms
<LISTING><H4>
_CADAC   =                  500
</H4></LISTING>
;;#f#<*********************************************************************/

int action_cadac(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  long 	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }
 
  if(Pt_acq!=NULL)
    {
      if(value>=0) 
	Pt_acq->cadac=value;
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  return(OK);
}
 

/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_nbfen
------------------

-Purpose:
Sets the number of windows used in the observation. It cannot be superior to
40 (which is the limit)
-Prototype:
int action_nbfen(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Example:
4 windows of data will be used in the observation
<LISTING><H4>
_NBFEN   =                    4
</H4></LISTING>
;;#f#<*********************************************************************/

int action_nbfen(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  char message[TAILLE_MES];
  long	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }
    
  sprintf(message,"valeur nfen: %i",value);


  if(Pt_acq!=NULL)
    {
      if( (value>0)&&(value<=NB_MAX_FEN) )
	{
	  Pt_acq->nbfen=value;

	  Tb_header_princ[Pt_enrg->nb_li_princ].indice=_NBWIN;
	  Tb_header_princ[Pt_enrg->nb_li_princ].type=TYPE_LONG;
	  Tb_header_princ[Pt_enrg->nb_li_princ].fenetre=NOTH;
	  Tb_header_princ[Pt_enrg->nb_li_princ].value.type_long=value;
	  Pt_enrg->nb_li_princ++;
	}
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  return(OK);
}


/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_nbacq
------------------

-Purpose:
Sets the number of acquisition that the system will do (depends on the
CTRAC value).

-Prototype:
int action_nbacq(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Example:
To perform 10 acquisitions:
<LISTING><H4>
_NBACQ   =                   10
</H4></LISTING>
;;#f#<*********************************************************************/

int action_nbacq(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char 	cval[TAILLE_VALU];
  long	value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  if(Pt_acq!=NULL)
    {
      if(value>=0)
	Pt_acq->nbacq=value;
      else
	{
	  the_errno.code=E_DECODE_SEM;
	  return(ERROR);
	}
    }
  return(OK);
}

action_wavlt(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char cval[TAILLE_VAL];
  char message[TAILLE_MES];
  int  ind=extrait_indice(ligne);
  double value=atof(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_float(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  (Pt_enrg->tab_phy).wavelnth[ind]=value;



  Tb_header_princ[Pt_enrg->nb_li_princ].indice=_WAVELNTH;
  Tb_header_princ[Pt_enrg->nb_li_princ].type=TYPE_DOUBLE;
  Tb_header_princ[Pt_enrg->nb_li_princ].fenetre=ind;
  Tb_header_princ[Pt_enrg->nb_li_princ].value.type_double=value;
  Pt_enrg->nb_li_princ++;


  return(OK);
	
}

action_nrbuf(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char cval[TAILLE_VAL];
  char message[TAILLE_MES];
  int  ind=extrait_indice(ligne);
  long value=atol(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU));
  
  if(ERROR==is_number(strncpy(cval,&ligne[DEBUT_VAL],TAILLE_VALU)))
    {
      the_errno.code=E_DECODE_SEM;
      return(ERROR);
    }

  (Pt_enrg->tab_phy).nrbuf[ind]=value;

  Tb_header_princ[Pt_enrg->nb_li_princ].indice=_NRBUF;
  Tb_header_princ[Pt_enrg->nb_li_princ].type=TYPE_LONG;
  Tb_header_princ[Pt_enrg->nb_li_princ].fenetre=ind;
  Tb_header_princ[Pt_enrg->nb_li_princ].value.type_long=value;
  Pt_enrg->nb_li_princ++;



  return(OK);
	
}



action_cmp_desc(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char cval[TAILLE_VAL];
  char message[TAILLE_MES];
  int taille_ligne;


  taille_ligne=strlen(ligne)-TAILLE_MCLE-TAILLE_SEPA;
  sprintf(message,"taille : %i",taille_ligne);


  taille_ligne=min(taille_ligne,TAILLE_VAL);
  sprintf(message,"min taille : %i",taille_ligne);


  Tb_header_princ[Pt_enrg->nb_li_princ].indice=_CMP_DESC;
  Tb_header_princ[Pt_enrg->nb_li_princ].type=TYPE_CHFITS;
  Tb_header_princ[Pt_enrg->nb_li_princ].fenetre=NOTH;
  strncpy(Tb_header_princ[Pt_enrg->nb_li_princ].value.type_chfits,\
	  &ligne[DEBUT_VAL],taille_ligne);

  Pt_enrg->nb_li_princ++;

  sprintf(message,"debut val: %c",ligne[DEBUT_VAL]);

  return(OK);	
}

action_binning(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
  char cval[TAILLE_VAL];
  char message[TAILLE_MES];
  int taille_ligne;


  taille_ligne=strlen(ligne)-TAILLE_MCLE-TAILLE_SEPA;
  sprintf(message,"taille : %i",taille_ligne);


  taille_ligne=min(taille_ligne,TAILLE_VAL);
  sprintf(message,"min taille : %i",taille_ligne);


  Tb_header_princ[Pt_enrg->nb_li_princ].indice=_BINNING;
  Tb_header_princ[Pt_enrg->nb_li_princ].type=TYPE_CHFITS;
  Tb_header_princ[Pt_enrg->nb_li_princ].fenetre=NOTH;
  strncpy(Tb_header_princ[Pt_enrg->nb_li_princ].value.type_chfits,\
	  &ligne[DEBUT_VAL],taille_ligne);

  Pt_enrg->nb_li_princ++;

  sprintf(message,"debut val: %c",ligne[DEBUT_VAL]);

  return(OK);
	
}

/*********************************************************************;;#f#>
------------------
HEAD_OF_FUNCTION:action_end
------------------

-Purpose:
 That's the last keyword which must appear at the bottom of the 
configuration file.
It returns a different values so that the search for other keyword stops.
-Prototype:
int action_end(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam);
-Parameters:
char	*ligne :
Tb_enrg	*Pt_enrg :
Tb_visu	*Pt_visu :
Tb_acq	*Pt_acq :
Tb_ctrl_cam *Pt_cam :
-Example:
<LISTING><H4>
_END     =                    0
</H4></LISTING>

;;#f#<*********************************************************************/

int action_end(ligne,Pt_enrg,Pt_visu,Pt_acq,Pt_cam)
char	*ligne;
Tb_enrg	*Pt_enrg;
Tb_visu	*Pt_visu;
Tb_acq	*Pt_acq;
Tb_ctrl_cam *Pt_cam;

{
	return(DECODE_FIN);
}










