/* Written by Stephen Levine @ the US Naval Observatory, Flagstaff */
/* Copyright USNO, all rights reserved */

#ifdef STANDALONE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "./tdefs.h"
#include "./flagbits.h"
#include "./Sla_bit/slalib.h"
#include "./Sla_bit/slamac.h"
#define ABS(x)   (((x) < 0) ? -(x) : (x))
#define MAX(a,b) (((a) < (b)) ? (b) : (a))
#define MIN(a,b) (((a) > (b)) ? (b) : (a))
#define GZPATH "/usr/bin/gzip"
#define GZSUF  "gz"
#define leave(x) {fprintf (stderr, "Exiting from %s\n", sbrtnname); exit(x);}
#else
#include "../Incl/hdrs.h"
#endif

#define S_GSC(x)  (sprintf(x,"%04d-%05d-%01d",sc->catid[0][i],\
                   (sc->catid[1][i])/10,(sc->catid[1][i])%10))

#define S_USN(x)  (sprintf(x,"%04d-%07d",sc->catid[0][i], sc->catid[1][i]))

#define HCOLTST(ci,cb,nc,a1,a2) if ((ci)&(cb)) { \
                                 sprintf (hdstr0+hsidx0,nc,a1); \
                                 sprintf (hdstr1+hsidx1,nc,a2); \
                                 hsidx0 = strlen(hdstr0); \
                                 hsidx1 = strlen(hdstr1); \
                               }

#define DCOLTST(ci,cb,nc,a1) if ((ci)&(cb)) { \
                                 sprintf (dtstr0+dsidx0, nc, a1); \
                                 dsidx0 = strlen(dtstr0); \
                               }

/* TITLE: wr_starcat - write out the contents of a STARCAT structure */
int wr_starcat (STARCAT *sc, PIXREGION *pixr, char *ofile)
{
  char *sbrtnname={"wr_starcat"};

  char          hdstr0[SLOGLINE], hdstr1[SLOGLINE], dtstr0[SLOGLINE];
  char          dsgn, syscmd[256], vtmp[16], sv1[8];
  int           fmt, srtkey, epkey, mergbits;
  int           i, j, k, rah, ram, decd, decm;
  int           hsidx0, hsidx1, dsidx0;
  unsigned long *skey;
  double        ras, decs, *dtoctr, *bmr;
  double        va[2], vat, *a2ep;
  FILE          *fo;

  fmt    = ( ((pixr->catal & CAT4_FMT) >> SHFT_CAT4_FMT) & (MASK_CAT4_FMT) );
  srtkey = ( ((pixr->catal & CAT4_SRT) >> SHFT_CAT4_SRT) & (MASK_CAT4_SRT) );
  epkey  = ( ((pixr->catal & CAT4_EPO) >> SHFT_CAT4_EPO) & (MASK_CAT4_EPO) );

  mergbits = sc->colbits & pixr->colbits;
  if (pixr->colbits & CB_RADIUS) mergbits |= CB_RADIUS; /* turn on DistToCtr */

/* Open output file */
  if ((fo = fopen(ofile, "w")) == (FILE *)NULL){
#ifdef DEBUG
#ifndef STANDALONE
    fprintf(stderr, "PIX%06.6d - ", requestid);
#endif
    fprintf (stderr,"%s: failed to fopen %s for writing\n", sbrtnname, ofile);
#endif
    leave ( -1 );
  }

/* print out search info */
  srchinfo (fo, sc->catname, pixr, sc->nstars);

/* Column Headers */
  bzero ( (void *)hdstr0, SLOGLINE );
  bzero ( (void *)hdstr1, SLOGLINE );

  hsidx0 = 0;
  hsidx1 = 0;

 /* id column */
  HCOLTST(mergbits,CB_CATID0,"%12.12s|","id","nd");

 /* RA & DEC */
  if (mergbits & CB_RADEC) {
    if (fmt == 0) {
      HCOLTST(mergbits,CB_RADEC,"%13.13s|","RA", "hh mm ss");
      HCOLTST(mergbits,CB_RADEC,"%13.13s|","DEC","dd mm ss");
    } else if (fmt == 1) {
      HCOLTST(mergbits,CB_RADEC,"%13.13s|","RA", "hh:mm:ss");
      HCOLTST(mergbits,CB_RADEC,"%13.13s|","DEC","dd:mm:ss");
    } else if (fmt == 2) {
      HCOLTST(mergbits,CB_RADEC,"%11.11s|","RA", "hh.hhh");
      HCOLTST(mergbits,CB_RADEC,"%11.11s|","DEC","dd.ddd");
    } else if (fmt == 3) {
      HCOLTST(mergbits,CB_RADEC,"%11.11s|","RA", "ddd.ddd");
      HCOLTST(mergbits,CB_RADEC,"%11.11s|","DEC","dd.ddd");
    }
  }

 /* Sigma RA & DEC */
  HCOLTST(mergbits,CB_SRADEC,"%3.3s|","sra","mas");
  HCOLTST(mergbits,CB_SRADEC,"%3.3s|","sde","mas");
    
 /* Epoch */
  if (mergbits & CB_EPOCH) {
    if (epkey == 0) {
      HCOLTST(mergbits,CB_EPOCH,"%14.14s|","Epoch","yyyy.yyyy");
    } else if (epkey == 1) {
      HCOLTST(mergbits,CB_EPOCH,"%14.14s|","Epoch","jd");
    } else {
      HCOLTST(mergbits,CB_EPOCH,"%14.14s|","Epoch","yyyy.yyyy");
    }
  }

 /* Proper Motion */
  HCOLTST(mergbits,CB_MU,"%6.6s|","MuRA", "mas/yr");
  HCOLTST(mergbits,CB_MU,"%6.6s|","MuDEC","mas/yr");

 /* Mu Probability */
  HCOLTST(mergbits,CB_MUPROB,"%6.6s|","MuProb","nd");

 /* Sigma Mu */
  HCOLTST(mergbits,CB_SMU,"%6.6s|","sMuRA","mas/yr");
  HCOLTST(mergbits,CB_SMU,"%6.6s|","sMuDE","mas/yr");

 /* RA/DEC Fit Sigma */
  HCOLTST(mergbits,CB_SFIT,"%6.6s|","sFitRA","mas");
  HCOLTST(mergbits,CB_SFIT,"%6.6s|","sFitDE","mas");

 /* # pts in Fit */
  HCOLTST(mergbits,CB_FITPTS,"%6.6s|","NFitPt","nd");

 /* Error column */
  HCOLTST(mergbits,CB_ERR,"%6.6s|","ErrFlg","nd");

 /* Flag byte */
  HCOLTST(mergbits,CB_FLG,"%6.6s|","Flags","nd");

 /* Multiple columns of magnitude etc. */
  if ( sc->nmagcol > 0 ) {
    for (i = 0; i < sc->nmagcol; i++) {
      HCOLTST(mergbits,CB_MAG,   "%7.7s|",sc->magid[i], "mag");
      HCOLTST(mergbits,CB_SMAG,  "%6.6s|","Sigma",      "mag");
      HCOLTST(mergbits,CB_MFLG,  "%6.6s|","MagFlg",     "nd");
      HCOLTST(mergbits,CB_FLDID, "%5.5s|","FldID",      "nd");
      HCOLTST(mergbits,CB_SG,    "%3.3s|","S/G",        "nd");
      HCOLTST(mergbits,CB_XYRES, "%8.8s|","XResid",     "arcsec");
      HCOLTST(mergbits,CB_XYRES, "%8.8s|","YResid",     "arcsec");
      HCOLTST(mergbits,CB_PLTIDX,"%10.10s|","PltIdx",   "nd");
    }
  }

 /* Xi & Eta */
  HCOLTST(mergbits,CB_XIETA,"%12.12s|","Xi", "arcsec");
  HCOLTST(mergbits,CB_XIETA,"%12.12s|","Eta","arcsec");

 /* Ecliptic Coords */
  HCOLTST(mergbits,CB_ELB,"%12.12s|","Ecl_L", "ddd.ddd");
  HCOLTST(mergbits,CB_ELB,"%12.12s|","Ecl_B", "ddd.ddd");

 /* Galactic Coords */
  HCOLTST(mergbits,CB_GLB,"%12.12s|","Gal_L", "ddd.ddd");
  HCOLTST(mergbits,CB_GLB,"%12.12s|","Gal_B", "ddd.ddd");

 /* X, Y Coords */
  HCOLTST(mergbits,CB_XY,"%12.12s|","X", "mm|pix");
  HCOLTST(mergbits,CB_XY,"%12.12s|","Y", "mm|pix");

 /* Measured X, Y Coords */
  HCOLTST(mergbits,CB_XYM,"%12.12s|","Xm", "mm|pix");
  HCOLTST(mergbits,CB_XYM,"%12.12s|","Ym", "mm|pix");

 /* Distance from Field Center */
  HCOLTST(mergbits,CB_RADIUS,"%7.7s|","DistCtr", "arcsec");

 /* Estimated Magnitude (if requested) */
  if (pixr->catal & CAT2_MAG) {	/* any estimated magnitudes? */
    if      (pixr->catal & CAT2_MAG_U) sprintf (sv1, "%s", "U");
    else if (pixr->catal & CAT2_MAG_B) sprintf (sv1, "%s", "B");
    else if (pixr->catal & CAT2_MAG_V) sprintf (sv1, "%s", "V");
    else if (pixr->catal & CAT2_MAG_R) sprintf (sv1, "%s", "R");
    else if (pixr->catal & CAT2_MAG_I) sprintf (sv1, "%s", "I");
    else if (pixr->catal & CAT2_MAG_J) sprintf (sv1, "%s", "J");
    else if (pixr->catal & CAT2_MAG_K) sprintf (sv1, "%s", "K");
    else if (pixr->catal & CAT2_MAG_L) sprintf (sv1, "%s", "L");
    else                               sprintf (sv1, "%s", "UNK");
  }
  HCOLTST(mergbits,CB_ESTMAG,"%7.7s|","EstMag",sv1);

  fprintf (fo, "%s\n", hdstr0);
  fprintf (fo, "%s\n", hdstr1);

/* Data Block */

/* compute distance to center in arcseconds */
  if ( mergbits & CB_RADIUS ) {
    dtoctr = (double *)malloc((unsigned)(sizeof(double) * sc->nstars));
    for (i = 0; i < sc->nstars; i++){
      dtoctr[i] = slaDsep (pixr->ra,pixr->dec,sc->ra[i],sc->dec[i])
	*DR2D*3600.0;
    }
  }

/* generate the output sorting index */
  skey = (unsigned long *)
    malloc((unsigned)(sizeof(unsigned long) * sc->nstars));
  for (i = 0; i < sc->nstars; i++) { skey[i] = (unsigned long)i + 1; }
    
  cat_sort (sc, pixr, mergbits, dtoctr, skey); 

  for (j = 0; j < sc->nstars; j++) {
    i = (int) skey[j] - 1;

    bzero ( (void *)dtstr0, SLOGLINE );
    dsidx0 = 0;

  /* ID */
    if (mergbits & CB_CATID0) {
      bzero((void *) vtmp, 16);
      if (mergbits & CB_CATID1) {
        if (sc->catid[1][i] != 0) { 
	  if (sc->fitpts[i] == 0) { S_GSC (vtmp); } /* FitsPts == 0 -> Tyc2 */
	  else {                    S_USN (vtmp); } /* else zone+rec# */
        } else {  	            sprintf (vtmp, "%d", sc->catid[0][i]); 
	}
      } else {                      sprintf (vtmp, "%d", sc->catid[0][i]); }
      DCOLTST(mergbits,CB_CATID0,"%12s ",vtmp);
    }

  /* RA + DEC */
    if (mergbits & CB_RADEC) {
      if (fmt == 0) {
	hd2hdms ( sc->ra[i]*DR2D/15.0,  &rah,  &ram,  &ras  );
	DCOLTST(mergbits,CB_RADEC,"%02d ",rah);
	DCOLTST(mergbits,CB_RADEC,"%02d ",ram);
	DCOLTST(mergbits,CB_RADEC,"%07.4lf ",ras);
	
	hd2hdms ( ABS(sc->dec[i]*DR2D), &decd, &decm, &decs );
	if ( sc->dec[i] < 0 ) {dsgn = '-';}
	else              {dsgn = '+';}
	DCOLTST(mergbits,CB_RADEC,"%c",      dsgn);
	DCOLTST(mergbits,CB_RADEC,"%02d ",   decd);
	DCOLTST(mergbits,CB_RADEC,"%02d ",   decm);
	DCOLTST(mergbits,CB_RADEC,"%06.3lf ",decs);
	
      } else if (fmt == 1) {
	hd2hdms ( sc->ra[i]*DR2D/15.0,  &rah,  &ram,  &ras  );
	DCOLTST(mergbits,CB_RADEC,"%02d:",rah);
	DCOLTST(mergbits,CB_RADEC,"%02d:",ram);
	DCOLTST(mergbits,CB_RADEC,"%07.4lf ",ras);
	
	hd2hdms ( ABS(sc->dec[i]*DR2D), &decd, &decm, &decs );
	if ( sc->dec[i] < 0 ) {dsgn = '-';}
	else                  {dsgn = '+';}
	DCOLTST(mergbits,CB_RADEC,"%c",      dsgn);
	DCOLTST(mergbits,CB_RADEC,"%02d:",   decd);
	DCOLTST(mergbits,CB_RADEC,"%02d:",   decm);
	DCOLTST(mergbits,CB_RADEC,"%06.3lf ",decs);
	
      } else if (fmt == 2) {
	DCOLTST(mergbits,CB_RADEC,"%011.8lf ", sc->ra[i]*DR2D/15.0);
	if ( sc->dec[i] < 0 ) {
	  dsgn = '-';
	  DCOLTST(mergbits,CB_RADEC,"%011.7lf ", sc->dec[i]*DR2D);
	} else                {
	  dsgn = '+';
	  DCOLTST(mergbits,CB_RADEC,"%c",      dsgn);
	  DCOLTST(mergbits,CB_RADEC,"%010.7lf ", sc->dec[i]*DR2D);
	}
	
      } else if (fmt == 3) {
	DCOLTST(mergbits,CB_RADEC,"%011.7lf ", sc->ra[i]*DR2D);
	if ( sc->dec[i] < 0 ) {
	  dsgn = '-';
	  DCOLTST(mergbits,CB_RADEC,"%011.7lf ", sc->dec[i]*DR2D);
	} else                {
	  dsgn = '+';
	  DCOLTST(mergbits,CB_RADEC,"%c",      dsgn);
	  DCOLTST(mergbits,CB_RADEC,"%010.7lf ", sc->dec[i]*DR2D);
	}
	
      } else {
	hd2hdms ( sc->ra[i]*DR2D/15.0,  &rah,  &ram,  &ras  );
	DCOLTST(mergbits,CB_RADEC,"%02d ",rah);
	DCOLTST(mergbits,CB_RADEC,"%02d ",ram);
	DCOLTST(mergbits,CB_RADEC,"%07.4lf ",ras);
	
	hd2hdms ( ABS(sc->dec[i]*DR2D), &decd, &decm, &decs );
	if ( sc->dec[i] < 0 ) {dsgn = '-';}
	else                  {dsgn = '+';}
	DCOLTST(mergbits,CB_RADEC,"%c",      dsgn);
	DCOLTST(mergbits,CB_RADEC,"%02d ",   decd);
	DCOLTST(mergbits,CB_RADEC,"%02d ",   decm);
	DCOLTST(mergbits,CB_RADEC,"%06.3lf ",decs);
      }
    }

 /* Sigma RA & DEC */
    DCOLTST(mergbits,CB_SRADEC,"%3d ", (int)sc->sra[i]);
    DCOLTST(mergbits,CB_SRADEC,"%3d ", (int)sc->sdec[i]);
    
 /* Epoch */
    DCOLTST(mergbits,CB_EPOCH,"%14.6lf ",sc->epoch[i]);

 /* Proper Motion */
    DCOLTST(mergbits,CB_MU,"%6d ",sc->mura[i]);
    DCOLTST(mergbits,CB_MU,"%6d ",sc->mudec[i]);

 /* Mu Probability */
    DCOLTST(mergbits,CB_MUPROB,"%6d ",sc->muprob[i]);

 /* Sigma Mu */
    DCOLTST(mergbits,CB_SMU,"%6d ",sc->smura[i]);
    DCOLTST(mergbits,CB_SMU,"%6d ",sc->smudec[i]);

 /* RA/DEC Fit Sigma */
    DCOLTST(mergbits,CB_SFIT,"%6d ",sc->srafit[i]);
    DCOLTST(mergbits,CB_SFIT,"%6d ",sc->sdecfit[i]);

 /* # pts in Fit */
    DCOLTST(mergbits,CB_FITPTS,"%6d ",sc->fitpts[i]);
    
 /* Error column */
    DCOLTST(mergbits,CB_ERR,"%6d ",sc->err[i]);

 /* Flag byte */
    DCOLTST(mergbits,CB_FLG,"%6c ",sc->flg[i]);

 /* Multiple columns of magnitude etc. */
    if ( sc->nmagcol > 0 ) {
      for (k = 0; k < sc->nmagcol; k++) {
	DCOLTST(mergbits,CB_MAG,   "%7.3lf " ,sc->mag[k][i]);
	DCOLTST(mergbits,CB_SMAG,  "%6.2lf " ,sc->smag[k][i]);
	DCOLTST(mergbits,CB_MFLG,  "%6d "    ,sc->mflg[k][i]);
	DCOLTST(mergbits,CB_FLDID, " %04.4d ",sc->fldid[k][i]);
	DCOLTST(mergbits,CB_SG,    "%3d "    ,sc->sg[k][i]);
	DCOLTST(mergbits,CB_XYRES, "%8.2lf " ,sc->xres[k][i]);
	DCOLTST(mergbits,CB_XYRES, "%8.2lf " ,sc->yres[k][i]);
	DCOLTST(mergbits,CB_PLTIDX,"%10d "   ,sc->pltidx[k][i]);
      }
    }

 /* Xi & Eta */
    DCOLTST(mergbits,CB_XIETA,"%12.3lf ",sc->xi[i]*DR2AS);
    DCOLTST(mergbits,CB_XIETA,"%12.3lf ",sc->eta[i]*DR2AS);
    
 /* Ecliptic Coords */
    DCOLTST(mergbits,CB_ELB,"%12.7lf ",sc->el[i]*DR2D);
    DCOLTST(mergbits,CB_ELB,"%12.7lf ",sc->eb[i]*DR2D);

 /* Galactic Coords */
    DCOLTST(mergbits,CB_GLB,"%12.7lf ",sc->gl[i]*DR2D);
    DCOLTST(mergbits,CB_GLB,"%12.7lf ",sc->gb[i]*DR2D);

 /* X, Y Coords */
    DCOLTST(mergbits,CB_XY,"%12.7lf ",sc->x[i]);
    DCOLTST(mergbits,CB_XY,"%12.7lf ",sc->y[i]);

 /* Measured X, Y Coords */
    DCOLTST(mergbits,CB_XYM,"%12.7lf ",sc->xm[i]);
    DCOLTST(mergbits,CB_XYM,"%12.7lf ",sc->ym[i]);

 /* Distance from Field Center */
    DCOLTST(mergbits,CB_RADIUS,"%7.1lf ", dtoctr[i]);

 /* Estimated Magnitude */
    DCOLTST(mergbits,CB_ESTMAG,"%7.3lf ", sc->estmag[i]);

    fprintf (fo, "%s\n", dtstr0);
  }

  fclose (fo);

  if (pixr->gzflg & GZF_GZIP) { /* gzip the file? 0=n,1=y */
    sprintf (syscmd, "%s %s", GZPATH, ofile);
    system(syscmd);
  }

  return ( 0 );
}
