/* Written by Stephen Levine @ the US Naval Observatory, Flagstaff */
/* Copyright USNO, all rights reserved */

/*

(Message inbox:527 + 531)
Return-Path: dgm@nofs.navy.mil

Date: Thu, 18 Apr 2002 10:29:30 -0700
Date: Thu, 2 May 2002 05:58:16 -0700

Subject: USNO-B FYI / USNO-B0.4

The current USNO-B generation pass is making files according to the
following format.

      (1)        ra 2000
      (2)        spd 2000
      (3)        iPVVVVUUUU
      (4)        jMRQyyyxxx
      (5)        keeevvvuuu

      (6)        FB
      (7)        FR
      (8)        SB
      (9)        SR
     (10)        xN

     (11)        FB x,y resid (CrrrrRRRR)
     (12)        FR x,y resid
     (13)        SB x,y resid
     (14)        SR x,y resid
     (15)        xN x,y resid

     (16)        FB index
     (17)        FR index
     (18)        SB index
     (19)        SR index
     (20)        xN index

     ra2000 is J2000 RA in units of 1/100 arcsec (0 - 129,599,999)

     spd2000 is J2000 South Polar Distance (==Dec+90) in units of
           1/100 arcsec (0 - 64,800,000)

     UUUU is proper motion in RA angle, not time, in units of 2 mas/year
           with an offset of 5000 added.  -5000 to 4999 mas/year get
           mapped into 0 - 9999. (= mu[arcsec/yr]*1000/2 + 5000)

     VVVV is proper motion in SPD in units of 2 mas/year.  Encoded as UUUU.
	   (= mu[arcsec/yr]*1000/2 + 5000)

     P is the motion probability (0-9)

     M is the number of points in motion fit (2-5)
       0 = Tycho2 star
       1 = single object detection - saved in separate rejected object file

     Q is error in RA fit (sigma_x) in units of 0.1 arcsec (0 - 9)

     R is error in SPD fit (sigma_y) in units of 0.1 arcsec (0 - 9)

     i = 1 if object is in a known proper motion catalogue

     j = 1 if object is close to a Diffraction Spike / Tycho Star

     k = 1 if object correlates with YS4.0

     xxx is error of RA motion in units of 1 mas/year (0 - 999)

     yyy is error of SPD motion in units of 1 mas/year (0 - 999)

     eee is mean epoch of observations in units of 0.1 year since 1950.0.
           Example: 1996.1 would appear as 461 (0 - 999)
           (= (year - 1950) * 10)

     uuu is error of RA coordinate in units of 1 mas (0 - 999)

     vvv is error if SPD coordinate in units of 1 mas (0 - 999)

The format for the First Blue (FB), First Red (FR), Second Blue (SB),
Second Red (SR), and N (xN) observations are identical.

          GGSFFFMMMM

      GG is the star/galaxy index (0 - 11), high(?) numbers mean star-like.
         0 = fuzzed out (0-3 = probably NOT a star)
	 11 = just like PSF (8-11 = probably IS a star)
	 19 = no S/G computed
         For all N surveys, GG==19.

         At the moment, all AO+xN plates GG == 19

      S is the survey
          0 = se = POSS-I O       (103aO)
          1 = so = POSS-I E       (103aE)
          2 = sj = POSS-II J      (IIIaJ)
          3 = sf = POSS-II F      (IIIaF)
          4 = sb = SERC-J         (IIIaJ)
          5 = sr = ESO-R/SERC-ER  (IIIaF)
          6 = ao = AAO-R          (IIIaF)
          7 = sn = POSS-II N      (IV-N)
          8 = an = AAO-N          (IV-N)
          9 = xn = AAO images of missed POSS-II N fields (IV-N)

      FFF is the field number in that survey (1 - 937)

      MMMM is calibrated magnitude in units of 0.01 magnitude.  Magnitudes
         outside the range 1 - 2499 are uncalibrated and refer to errors
         sensed by the PMM realtime software.
	 ( = mag * 100, range = 0.01 to 24.99 )

The format for the 5 x,y residual entries are
	CrrrrRRRR
where
	rrrr = 100 * y resid (in arcsec) + 5000
	RRRR = 100 * x resid (in arcsec) + 5000
	   The range is there for -50.00 to +49.99 arcsec.
        C = Flag for the photometric calibration source (range 0-9)
	     0 = calibrated from the bright photometric stds on this plate
	     1 = calibrated from the faint photometric stds 0 plate away 
	                                  (ie on this plate)
	     2 = faint photometric stds 1 plate away (ie on overlap plate)
	     3 = faint photometric stds 2 plate away (ie on overlap of 
	                                                   overlap plate)
	     etc.

The final 5 entries are the index of this observation in the raw data
files (i.e., you need to decode S and FFF as well as having access to
the Terabyte of raw PMM scans) just in case futher processing is needed
at some later time.  You can ignore these.

If observations are missing (i.e., P < 5), the appropriate line(s) are
set to zero.

Tycho-2 was copied from NSSDC and these data were inserted into the catalog.
The format of the Tycho-2 entries is the same length, and most are similar.

      ( 1)      ra2000
      ( 2)      spd2000
      ( 3)      VVVVUUUU
      ( 4)      file (0-19, 20=suppl_1, 21=suppl_2) from NSSDC
      ( 5)      sequential index into the file
      ( 6)      MMMM - O magnitude as computed from the Tycho Bt+Vt
      ( 7)             E
      ( 8)             J
      ( 9)             F
      (10)             N
      (11)      TYC1 = GSC region #
      (12)      10 * TYC2 + TYC3 (TYC2 = running# in region, TYC3 = component)
      (13) - (15) zero
      (16) - (20) zero

You need only a single test to sense a Tycho-2 entry using (4) since
USNO-B stars are guaranteed to have 1 <= M <= 5, so a value for (4)
less than 100,000,000 must be a Tycho-2 star.

-Dave

*/

/* Item    IDX    - SIZE  - Description                - UNITS            */
/* ----    -----    ----    -----------                  -----            */
/* ra2000      0    1 INT   J2000 RA                     int 0.01 arcsec  */
/* spd2000     1    1 INT   J2000 South Polar Distance   int 0.01 arcsec  */
/*                                (==Dec+90)                              */
/* PVVVVUUUU   2    1 INT   proper motions (see above)   int mas/yr       */
/* MRQyyyxxx   3    1 INT   errors (posn fit, mu's)      int 0.1 arcsec   */
/*                                                       int mas/yr       */
/* eeevvvuuu   4    1 INT   epoch + errors (posn)        int 0.1 yr       */
/*                                                       int 0.1 arcsec   */
/* FB          5    1 INT   GGSFFFMMMM - s/g+fld+mags    int 0.01 mag     */
/* FR          6    1 INT   GGSFFFMMMM - s/g+fld+mags    int 0.01 mag     */
/* SB          7    1 INT   GGSFFFMMMM - s/g+fld+mags    int 0.01 mag     */
/* SR          8    1 INT   GGSFFFMMMM - s/g+fld+mags    int 0.01 mag     */
/* xN          9    1 INT   GGSFFFMMMM - s/g+fld+mags    int 0.01 mag     */
/* FBx,y res. 10    1 INT   rrrrRRRR - x,y resid         int 0.01 arcsec  */
/* FRx,y res. 11    1 INT   rrrrRRRR - x,y resid         int 0.01 arcsec  */
/* SBx,y res. 12    1 INT   rrrrRRRR - x,y resid         int 0.01 arcsec  */
/* SRx,y res. 13    1 INT   rrrrRRRR - x,y resid         int 0.01 arcsec  */
/* xNx,y res. 14    1 INT   rrrrRRRR - x,y resid         int 0.01 arcsec  */
/* FB index   15    1 INT                                                 */
/* FR index   16    1 INT                                                 */
/* SB index   17    1 INT                                                 */
/* SR index   18    1 INT                                                 */
/* xN index   19    1 INT                                                 */

/* so, each record is (20 * 4 = 80) bytes long, and all values are integers */
/* stephen levine (sel@picacho) 06 may2002 */

/* Flag bits - 01234567 - */
/*  bit val meaning */
/*  0 = 1 = Proximity to a proper motion catalogue star (LTT,NLTT,Giclas etc)*/
/*  1 = 2 = Proximity to a Diffraction Spike / Tycho-2 star */
/*  2 = 4 = Is a YS4.0 star */
/*  3 = 8 = Is a salted in Tycho-2 star */

#ifdef STANDALONE
#include "./flagbits.h"
#include "./tdefs.h"
#else
#include "../../Incl/hdrs.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <limits.h>     /* definition of OPEN_MAX */
#include <unistd.h>

#ifdef STANDALONE2
#include "./Sla_bit/slalib.h"
#include "./Sla_bit/slamac.h"
#else
#include "/usr/local/include/slalib.h"
#include "/usr/local/include/slamac.h"
#endif

/* Generic Macros */
#define	ABS(x)	 (((x) < 0) ? -(x) : (x))
#define	MAX(a,b) (((a) < (b)) ? (b) : (a))
#define	MIN(a,b) (((a) > (b)) ? (b) : (a))

/* Numerical Constants */
#define PI	 3.141592653589793238462643e0
#define TWOPI    6.283185307179586476925286766559e0
#define HALFPI	 (0.5 * PI)
#define DEG2RAD  (PI / 180.0)
#define TORAD    (PI / 180.0)

#define NPLATES    5
#define NINTREC    20
#define NZMAX      1800		/* 0.1 * 1800 = 180 == All sky */
#define NACC       96
/* #define NCHUNK   1000000 */
#define NCHUNK     256000
#define CONVERT    (360000.0)	/* 3600.0 * 100.0 */
#define NHIST      25
#define MAXUBSTARS 250000 
/*#define MAXUBSTARS 2000000*/

#define leave(x) {fprintf (stderr, "Exiting from %s\n", sbrtnname); exit(x);}

/* pixel server path to usnob */
#ifdef V_USAF
#define BPATH "/cat/usnob/"
#else
#define BPATH "/ig7/sel/PixServe/Catalogues/USNOB/"
#endif

/*--------- Macros to go from Binary USNO-B Format to internal --------- */

/* RA/DEC in radians */
#define D_RA2K(x)     (x=(((double)(ub_buf[j][0]))/(360000.) * TORAD))
#define D_DEC2K(x)    (x=((((double)(ub_buf[j][1]))/(360000.) - 90.0) * TORAD))

/* MU_(RA/DEC) in mas/yr, MU_probability */
#define I_MURA2K(x)   (x=((ABS(ub_buf[j][2])%10000 - 5000)*2))
#define I_MUDEC2K(x)  (x=(((ABS(ub_buf[j][2])/10000)%10000 - 5000)*2))
/* #define I_MURA2K(x)   (x=(ABS(ub_buf[j][2])%10000 - 5000)) */
/* #define I_MUDEC2K(x)  (x=((ABS(ub_buf[j][2])/10000)%10000 - 5000)) */
#define I_MUPROB(x)   (x=((ABS(ub_buf[j][2])/100000000)%10))
#define B_MUCATLG(x)  (x|=(MAX(MIN(1,((ABS(ub_buf[j][2])/1000000000)%10)),0))<<0)

/* MU_RA/DEC Sigma mas/yr */
#define I_MURASIG(x)  (x=(ABS(ub_buf[j][3])%1000))
#define I_MUDECSIG(x) (x=((ABS(ub_buf[j][3])/1000)%1000))

/* RA/DEC Fit Sigma (stored in integer 0.1 arcsec, convert to mas) */
#define I_RAFSIG(x)   (x=(((ABS(ub_buf[j][3])/1000000)%10)*100))
#define I_DECFSIG(x)  (x=(((ABS(ub_buf[j][3])/10000000)%10)*100))

/* Number of points in the motion fit */
#define I_FITPTS(x)   (x=((ABS(ub_buf[j][3])/100000000)%10))
#define B_TYCPROX(x)  (x|=(MAX(MIN(1,((ABS(ub_buf[j][3])/1000000000)%10)),0))<<1)

/* Error in RA/DEC mas */
#define I_SIGRA(x)    (x=(ABS(ub_buf[j][4])%1000))
#define I_SIGDEC(x)   (x=((ABS(ub_buf[j][4])/1000)%1000))

/* Mean Epoch of Obs. */
#define D_MEANEP(x)   (x=(((double)((ABS(ub_buf[j][4])/1000000)%1000))/10.0 + 1950.0))
#define B_YS4CATLG(x) (x|=(MAX(MIN(1,((ABS(ub_buf[j][4])/1000000000)%10)),0))<<2)

/* Magnitudes,Fields,Star/Galaxy */
#define D_MAG(x,y)    (x=(((double)(ABS(y)%10000))/100.0))
#define I_FLD(x,y)    (x=((ABS(y)/10000)%10000))
#define I_SVY(x,y)    (x=((ABS(y)/10000000)%10)) /* (included in FLD) */
#define I_SG(x,y)     (x=((ABS(y)/100000000)%100))

#define D_FBMAG(x)    (D_MAG(x,ub_buf[j][5]))
#define D_FRMAG(x)    (D_MAG(x,ub_buf[j][6]))
#define D_SBMAG(x)    (D_MAG(x,ub_buf[j][7]))
#define D_SRMAG(x)    (D_MAG(x,ub_buf[j][8]))
#define D_xNMAG(x)    (D_MAG(x,ub_buf[j][9]))

#define I_FBFLD(x)    (I_FLD(x,ub_buf[j][5]))
#define I_FRFLD(x)    (I_FLD(x,ub_buf[j][6]))
#define I_SBFLD(x)    (I_FLD(x,ub_buf[j][7]))
#define I_SRFLD(x)    (I_FLD(x,ub_buf[j][8]))
#define I_xNFLD(x)    (I_FLD(x,ub_buf[j][9]))

#define I_FBSVY(x)    (I_SVY(x,ub_buf[j][5]))
#define I_FRSVY(x)    (I_SVY(x,ub_buf[j][6]))
#define I_SBSVY(x)    (I_SVY(x,ub_buf[j][7]))
#define I_SRSVY(x)    (I_SVY(x,ub_buf[j][8]))
#define I_xNSVY(x)    (I_SVY(x,ub_buf[j][9]))

#define I_FBSG(x)     (I_SG(x,ub_buf[j][5]))
#define I_FRSG(x)     (I_SG(x,ub_buf[j][6]))
#define I_SBSG(x)     (I_SG(x,ub_buf[j][7]))
#define I_SRSG(x)     (I_SG(x,ub_buf[j][8]))
#define I_xNSG(x)     (I_SG(x,ub_buf[j][9]))

/* x,y residuals */
#define D_XRESID(x,y) (x=( ((double)((ABS(y)%10000) - 5000)) / 100.0 ))
#define D_YRESID(x,y) (x=( ((double)((ABS(y)/10000)%10000 - 5000)) / 100.0 ))
#define I_MFLG(x,y)   (x=( (ABS(y)/100000000)%10 ))

#define D_FBXRES(x)   (D_XRESID(x,ub_buf[j][10]))
#define D_FRXRES(x)   (D_XRESID(x,ub_buf[j][11]))
#define D_SBXRES(x)   (D_XRESID(x,ub_buf[j][12]))
#define D_SRXRES(x)   (D_XRESID(x,ub_buf[j][13]))
#define D_xNXRES(x)   (D_XRESID(x,ub_buf[j][14]))

#define D_FBYRES(x)   (D_YRESID(x,ub_buf[j][10]))
#define D_FRYRES(x)   (D_YRESID(x,ub_buf[j][11]))
#define D_SBYRES(x)   (D_YRESID(x,ub_buf[j][12]))
#define D_SRYRES(x)   (D_YRESID(x,ub_buf[j][13]))
#define D_xNYRES(x)   (D_YRESID(x,ub_buf[j][14]))

#define I_FBMFLG(x)   (I_MFLG(x,ub_buf[j][10]))
#define I_FRMFLG(x)   (I_MFLG(x,ub_buf[j][11]))
#define I_SBMFLG(x)   (I_MFLG(x,ub_buf[j][12]))
#define I_SRMFLG(x)   (I_MFLG(x,ub_buf[j][13]))
#define I_xNMFLG(x)   (I_MFLG(x,ub_buf[j][14]))

/* Plate Indexes */
#define I_PLIDX(x,y)  (x=y)

#define I_FBPLIDX(x)  (I_PLIDX(x,ub_buf[j][15]))
#define I_FRPLIDX(x)  (I_PLIDX(x,ub_buf[j][16]))
#define I_SBPLIDX(x)  (I_PLIDX(x,ub_buf[j][17]))
#define I_SRPLIDX(x)  (I_PLIDX(x,ub_buf[j][18]))
#define I_xNPLIDX(x)  (I_PLIDX(x,ub_buf[j][19]))

/* Tycho2 related macros */
#define B_TYCHO2      ((ub_buf[j][3] < 100000000) ? 1 : 0)

#define I_TYCID(x)    {(x)[0]=ub_buf[j][10];(x)[1]=ub_buf[j][11];}
#define S_TYCID(x,y)  (sprintf(x,"%04d %05d %01d",y[0],y[1]/10,y[1]%10))
#define B_ISTYC(x)    (x|=(1<<3))

/*--------- Macros to go from internal to Binary USNO-B Format --------- */

/* RA/DEC in radians */
#define RA2K_A(x)     (ub_buf[j][0]=(int)((x)/TORAD*360000.))
#define DEC2K_A(x)    (ub_buf[j][1]=(int)(((x)/TORAD+90.0)*360000.))

/* ---------------------------------------------------------------------- */

int ub_nra, ub_ndec, ub_dzone[NZMAX+1], ub_frec[NACC+1], ub_nrec[NACC+1];
int ub_oldzone, ub_buf[NCHUNK][NINTREC], ub_nsav;
int ub_bhist[NHIST+1], ub_rhist[NHIST+1];

double ub_rfrst[2+1], ub_rlast[2+1], ub_dfrst[NZMAX+1], ub_dlast[NZMAX+1];
double ub_rcent, ub_dcent, ub_scale;
double ub_rac_ct, ub_decc_ct;	/* field center in catalogue epoch */
char   ub_eqstr[4], *b_base;

double UBMnRa, UBMxRa, UBMnDec, UBMxDec, UBMnBxx, UBMxBxx, UBMnBxy, UBMxBxy;
int    UBMaxpts;
int    UBPrimMag, UBClr0, UBClr1;

char   IFile[256];
int    OFileType;

int    ubnpts;
int    b_fd;
