/* Written by Stephen Levine @ the US Naval Observatory, Flagstaff */
/* Copyright USNO, all rights reserved */

#include "./ubsquare.h"

/* TITLE: ubc_saveit */
/* unpack 80 byte usnob record into internal structure (starcat) */

int ubc_saveit(int j, int dz, int rnum, PIXREGION *pixr, STARCAT *sc)
{
  char   flgbits = '\0';

  int    obid[2];
  int    rasig = 0, decsig = 0;
  int    mura2k = 0, mudec2k = 0, muprob = 0, murasig = 0, mudecsig = 0;
  int    rafitsig = 0, decfitsig = 0,  nfitpts = 0;
  int    flds[NPLATES], stargal[NPLATES];
  int    plateidx[NPLATES], mflg[NPLATES];
  int    i, jk, starinbox = 0;
  int    isg = -1;

  double ra2k = 0.0, dec2k = 0.0;
  double meanep = 0.0;
  double xresid[NPLATES], yresid[NPLATES];
  double mags[NPLATES];
  double axi = 0.0, aeta = 0.0, wra = 0.0, wdec = 0.0, wdra = 0.0, wddec = 0.0;
  double wdsgn = 1.0;

  double star_colors (char *, double, double, char *);

  /* see ubsquare for explanation of catalogue internal units */
  /* the units w/in this program will be radians, mas/yr, mags etc */

  D_RA2K(ra2k);			/* -> radians */
  D_DEC2K(dec2k);		/* -> radians */

  I_MURA2K(mura2k);		/* -> mas/yr */
  I_MUDEC2K(mudec2k);		/* -> mas/yr */

  D_FBMAG(mags[0]);		/* mag */
  D_FRMAG(mags[1]);		/* mag */
  D_SBMAG(mags[2]);		/* mag */
  D_SRMAG(mags[3]);		/* mag */
  D_xNMAG(mags[4]);		/* mag */

  if (B_TYCHO2) {		/* if a Tycho2 star */

    rasig    = decsig = 0;
    rafitsig = decfitsig = 0;
    nfitpts  = 0;
    muprob   = 0;
    murasig  = mudecsig = 0.0;
    meanep   = 2000.0;
    for (i = 0; i < NPLATES; i++) {
      mflg[i]     = 0;
      flds[i]     = 0;
      stargal[i]  = 11;
      plateidx[i] = 0;
      xresid[i]   = yresid[i] = 0.0;
    }
    B_ISTYC(flgbits);		/* set bit that says this is a Tycho star */
    I_TYCID(obid);
#if 0
    fprintf (stderr, "ub_saveit: ub_buf[10,11] = %d %d, obid[0,1] = %d %d\n",
	     ub_buf[j][10], ub_buf[j][11], obid[0], obid[1]);
#endif


  } else {			/* else, a PMM detection */

    obid[0] = dz;		/* zone number */
    obid[1] = rnum;		/* record number w/in zone */

    I_MUPROB(muprob);		/* 0-9 */
    B_TYCPROX(flgbits);		/* set bit for Tycho proximity */

    I_RAFSIG(rafitsig);		/* -> mas */
    I_DECFSIG(decfitsig);	/* -> mas */
    I_FITPTS(nfitpts);		/* # */
    B_MUCATLG(flgbits);		/* set bit for in prop. mot. catalogue */

    I_SIGRA(rasig);		/* mas */
    I_SIGDEC(decsig);		/* mas */

    D_MEANEP(meanep);		/* decimal year */
    B_YS4CATLG(flgbits);	/* set bit for in YS4.0 catalogue */

    I_MURASIG(murasig);		/* sigma mu ra */
    I_MUDECSIG(mudecsig);	/* sigma mu dec */

    I_FBFLD(flds[0]);		/* fld id */
    I_FRFLD(flds[1]);		/* fld id */
    I_SBFLD(flds[2]);		/* fld id */
    I_SRFLD(flds[3]);		/* fld id */
    I_xNFLD(flds[4]);		/* fld id */
    
    I_FBSG(stargal[0]);		/* Star/Galaxy # */
    I_FRSG(stargal[1]);		/* Star/Galaxy # */
    I_SBSG(stargal[2]);		/* Star/Galaxy # */
    I_SRSG(stargal[3]);		/* Star/Galaxy # */
    I_xNSG(stargal[4]);		/* Star/Galaxy # */

    D_FBXRES(xresid[0]);		/* 0.01 arcsec */
    D_FRXRES(xresid[1]);		/* 0.01 arcsec */
    D_SBXRES(xresid[2]);		/* 0.01 arcsec */
    D_SRXRES(xresid[3]);		/* 0.01 arcsec */
    D_xNXRES(xresid[4]);		/* 0.01 arcsec */

    D_FBYRES(yresid[0]);		/* 0.01 arcsec */
    D_FRYRES(yresid[1]);		/* 0.01 arcsec */
    D_SBYRES(yresid[2]);		/* 0.01 arcsec */
    D_SRYRES(yresid[3]);		/* 0.01 arcsec */
    D_xNYRES(yresid[4]);		/* 0.01 arcsec */

    I_FBMFLG(mflg[0]);		/* mag flag(s) */
    I_FRMFLG(mflg[1]);		/* mag flag(s) */
    I_SBMFLG(mflg[2]);		/* mag flag(s) */
    I_SRMFLG(mflg[3]);		/* mag flag(s) */
    I_xNMFLG(mflg[4]);		/* mag flag(s) */

    I_FBPLIDX(plateidx[0]);		/* Index into plate file */
    I_FRPLIDX(plateidx[1]);		/* Index into plate file */
    I_SBPLIDX(plateidx[2]);		/* Index into plate file */
    I_SRPLIDX(plateidx[3]);		/* Index into plate file */
    I_xNPLIDX(plateidx[4]);		/* Index into plate file */
  }  

  starinbox = 0;

  axi = 0.0;
  aeta = 0.0;

  wra = ra2k;
  wdec = dec2k;

/* ONLY if we are doing a Binary to Ascii Dump, force the starinbox on */
  if ( pixr->gzflg & GZF_FTYPE_DMP ) { starinbox = 1; }

/* check mag criteria */
  if ( mag_match ( mags[UBPrimMag], pixr->brtmag, pixr->fntmag ) ) {

/* check position criteria */
    if (pixr->catal & CAT3_EXT_1) { /* Rectangle */
      rd2ze(wra, wdec, pixr->ra, pixr->dec, &(axi), &(aeta), &jk);
	    
      if (axi  >= UBMnBxx && axi  <= UBMxBxx && 
	  aeta >= UBMnBxy && aeta <= UBMxBxy) starinbox = 1;
		
    } else if (pixr->catal & CAT3_EXT_2) { /* Circle */
      if (slaDsep(pixr->ra,pixr->dec,wra,wdec)<0.5*pixr->dra) {
	rd2ze(wra, wdec, pixr->ra, pixr->dec, &(axi), &(aeta), &jk);
	starinbox = 1;
      }
    } else if (pixr->catal & CAT3_EXT_3) { /* Dec Band */
      if (2.0 * ABS(wdec - pixr->dec) <= pixr->ddec) {
        rd2ze(wra, wdec, pixr->ra, pixr->dec, &(axi), &(aeta), &jk);
        starinbox = 1;
      }
    }
  }
  
  if ((ubnpts < UBMaxpts) && starinbox) {
    sc->catid[0][ubnpts] = obid[0];
    sc->catid[1][ubnpts] = obid[1];

    sc->ra[ubnpts]     = wra;
    sc->dec[ubnpts]    = wdec;
    sc->sra[ubnpts]    = (double)rasig;
    sc->sdec[ubnpts]   = (double)decsig;

    sc->mura[ubnpts]   = mura2k;
    sc->mudec[ubnpts]  = mudec2k;

    sc->muprob[ubnpts] = muprob;
    sc->smura[ubnpts]  = murasig;
    sc->smudec[ubnpts] = mudecsig;

    sc->srafit[ubnpts] = rafitsig;
    sc->sdecfit[ubnpts]= decfitsig;
    sc->fitpts[ubnpts] = nfitpts;

    sc->epoch[ubnpts]  = meanep;

    for (i = 0; i < sc->nmagcol; i++) {
      sc->mag[i][ubnpts]   = mags[i];
      sc->mflg[i][ubnpts]  = mflg[i];
      sc->fldid[i][ubnpts] = flds[i];
      sc->sg[i][ubnpts]    = stargal[i];

      if (stargal[i] == 0 && mags[i] == 0.0) sc->sg[i][ubnpts]  = 19;
      else                                   sc->sg[i][ubnpts]  = stargal[i];

      sc->xres[i][ubnpts]  = xresid[i];
      sc->yres[i][ubnpts]  = yresid[i];
      sc->pltidx[i][ubnpts]= plateidx[i];
    }

    /* 48 maps 00000000 to character 0 in the ASCII set */
    /*         00000001 to           1 */
    /*         00000010 to           2 etc */
    sc->flg[ubnpts] = (((int)flgbits<=9) ? ((char)((int)flgbits + 48)) :
       ((char)((int)flgbits + 55)));

    sc->xi[ubnpts]  = axi;
    sc->eta[ubnpts] = aeta;

    if (b_fd > 0) ubc_1rec_binwr(j); /* write out the binary record */

    ubnpts++;
  }

  return ( 0 );
}

/* TITLE: ubc_1rec_binwr - write out 1 binary record */
int ubc_1rec_binwr (int j)
{
  int i = -1;

  i = write (b_fd, (char *)(ub_buf[j]), 80);
  return ( i );
}
