/* Written by Stephen Levine @ the US Naval Observatory, Flagstaff */
/* Copyright USNO, all rights reserved */

#include "./ubsquare.h"
#include "./version.h"

#define HR2RAD   (PI / 12.0)    /* M_PI is in <math.h> */

#define CMDLARGS \
 "B:F:f:HhI:i:L:O:o:P:p:QS:s:Z:z:"

#define UBC_MAXBOX   4.0
#define UBC_MAXBND 360.0

/* TITLE: ubc_cmdline - parse USNO-B subset cmd line */

PIXREGION *ubc_cmdline ( int argc, char **argv )
{
  char *sbrtnname = {"ubc_cmdline"};

  extern char  *optarg;
  extern int   optind, opterr;

  int    c, i, iv1, glbflg = 0;
  double v1, height_max = UBC_MAXBOX,  width_max = UBC_MAXBND;

  PIXREGION *pixr;
  int j, nflds;

/* call usage arguments and exit if no args on cmd line */
  if (argc == 1) {
    ubc_usage2();
    exit (-1);
  }

/* echo the command line */
  fprintf (stderr, "%s: calling sequence -", sbrtnname);
  for (i = 0;  i < argc; i++) {
    fprintf (stderr, " %s", argv[i]);
  }
  fprintf (stderr, "\n");

  OFileType = 0 | CAT5_OFILE_ASC;		/* ASCII output by default */

  nflds = 1;
  pixr = (PIXREGION *)pixmalloc ((unsigned)(nflds * sizeof(PIXREGION)),
				 "ub_cmdline - pixr");

  for (j = 0; j < nflds; j++) {
    sprintf(pixr[j].fldnam, "%s", "Field");
    pixr[j].ra     = 0.0;	/* 0hr */
    pixr[j].dec    = 0.0;	/* 0deg */
    pixr[j].dra    = 0.0029089;	/* 10 arcmin */
    pixr[j].ddec   = 0.0029089;
    pixr[j].brtmag =    0.0;
    pixr[j].fntmag =  100.0;
    pixr[j].bmrmin = -100.0;
    pixr[j].bmrmax =  100.0;
    strcpy (pixr[j].clr,       "R2"); /* R_photo */
    strcpy (pixr[j].clr0m1[0], "B2"); /* B_photo */
    strcpy (pixr[j].clr0m1[1], "R2"); /* R_photo */
    sprintf(pixr[j].pname,"%s", "Tmp");
    pixr[j].ori   = 0;
    pixr[j].gzflg = 0;
    pixr[j].ticks = 0;

    pixr[j].catal = (CAT1_ID_USNOB|CAT3_EXT_1|CAT4_FMT_2|CAT4_SRT_NUL);
    
    pixr[j].surveys = 0;
    pixr[j].devtype = 0;
    pixr[j].cmark   = 0.0;
    strcpy(pixr[j].email, "none");
    
    pixr[j].equin    = 2000.0;	/* equinox */
    pixr[j].epoch    = 2000.0;	/* epoch */
    pixr[j].mumin    = 0.0;	/* mu  in mas/yr */
    pixr[j].mumax    = 10000.0;
    pixr[j].sposmin  = 0.0;	/* pos'n error in mas */
    pixr[j].sposmax  = 10000.0;
    pixr[j].smumin   = 0.0;	/* sigma mu  in mas/yr */
    pixr[j].smumax   = 10000.0;
    pixr[j].smagmin  = 0.0;	/* mag. error */
    pixr[j].smagmax  = 1.0;
    pixr[j].densmin  = 0.0;	/* density #/(sq.deg) */
    pixr[j].densmax  = 10000000.0;
    pixr[j].minsep   = 0.0;	/* min.separation between objs (radians)*/
    pixr[j].msdmag   = 100.0;	/* min.sep. mag limit between objs */
    pixr[j].minsg    = 0;	/* min. star/gal index */
    pixr[j].maxsg    = 11;	/* max. star/gal index */
    pixr[j].pixscale = 67.14/1000.*6.8*2.0; /*output scale arcsec/pix */

/*
    pixr[j].colbits  = 0 | 
      (CB_CATID0   | CB_CATID1
       | CB_RADEC  | CB_SRADEC 
       | CB_EPOCH
       | CB_MU     | CB_MUPROB | CB_SMU
       | CB_SFIT   | CB_FITPTS | CB_FLG
       | CB_MAG    | CB_SG     | CB_XIETA );
*/
    pixr[j].colbits = 0 | (CB_CATID0 | CB_CATID1
		 | CB_RADEC 
		 | CB_SRADEC 
		 | CB_EPOCH  
		 | CB_MU 
		 | CB_MUPROB 
		 | CB_SMU
		 | CB_SFIT 
		 | CB_FITPTS 
		 | CB_FLG
		 | CB_MAG 
		 | CB_MFLG 
		 | CB_FLDID 
		 | CB_SG 
		 | CB_SVY 
		 | CB_XYRES 
/*		 | CB_PLTIDX*/
		 | CB_XIETA 
		 );

  }

  while ((c = getopt(argc, argv, CMDLARGS)) != -1) {
    switch (c) {

    case 'B':			/* Field Name */
      for (j = 0; j < nflds; j++) 
	sprintf(pixr[j].fldnam, "%s%d", optarg, j);
      break;

    case 'F':
    case 'f':			/* Rect/Circ Flag */
      if        (!strcmp(optarg, "R")) { /* Rectangular Region */
	for (j = 0; j < nflds; j++) 
	  pixr[j].catal = ((pixr[j].catal & ~(CAT3_EXT)) | (CAT3_EXT_1));
      } else if (!strcmp(optarg, "C")) { /* Circular Region */
	for (j = 0; j < nflds; j++) 
	  pixr[j].catal = ((pixr[j].catal & ~(CAT3_EXT)) | (CAT3_EXT_2));
      } else if (!strcmp(optarg, "B")) { /* DEC Band */
	for (j = 0; j < nflds; j++) 
	  pixr[j].catal = ((pixr[j].catal & ~(CAT3_EXT)) | (CAT3_EXT_3));
      } else {
	fprintf (stderr, 
	   "%s: invalid option for -F, must be one of:\n", sbrtnname);
	fprintf (stderr, "  R = rectangular region\n");
	fprintf (stderr, "  C = circular region\n");
	fprintf (stderr, "  B = DEC Band\n");

	exit (-1);
      }
      break;

    case 'I':			/* Bright Mag */
      v1 = atof(optarg);
      if (v1 < -1 || v1 > 100) {
	fprintf (stderr, "%s: Magnitude range is -1 to 100\n", sbrtnname);
	exit ( -1 );
      }
      for (j = 0; j < nflds; j++) pixr[j].brtmag = v1;
      break;

    case 'i':			/* Faint Mag */
      v1 = atof(optarg);
      if (v1 < -1 || v1 > 100) {
	fprintf (stderr, "%s: Magnitude range is -1 to 100\n", sbrtnname);
	exit ( -1 );
      }
      for (j = 0; j < nflds; j++) pixr[j].fntmag = v1;
      break;

    case 'L':			/* Primary Mag sort color */
      if (!strcmp(optarg, "B1") ||
	  !strcmp(optarg, "R1") ||
	  !strcmp(optarg, "B2") ||
	  !strcmp(optarg, "R2") ||
	  !strcmp(optarg, "I2")) {
	for (j = 0; j < nflds; j++) strcpy(pixr[j].clr, optarg);
      } else {
	fprintf (stderr, "%s: Allowed Color Names are\n", sbrtnname);
	fprintf (stderr, "       B1 - First Epoch Blue\n");
	fprintf (stderr, "       R1 - First Epoch Red\n");
	fprintf (stderr, "       B2 - Second Epoch Blue\n");
	fprintf (stderr, "       R2 - Second Epoch Red\n");
	fprintf (stderr, "       I2 - Second Epoch Near-IR\n");
	exit ( -1 );
      }
      break;

    case 'O':			/* Output File */
      for (j = 0; j < nflds; j++) sprintf(pixr[j].pname, "%s", optarg);
      break;

    case 'P':			/* RA Position */
      v1 = atof(optarg);
      if (v1 < 0.0 || v1 > 24.0) {
	fprintf (stderr, "%s: RA out of range 0 to 24hrs\n", sbrtnname);
	exit (-1);
      }
      v1 *= HR2RAD;
      for (j = 0; j < nflds; j++) pixr[j].ra = v1;
      break;

    case 'p':			/* Dec Position */
      v1 = atof(optarg);
      if (v1 < -90.0 || v1 > 90.0) {
	fprintf (stderr, "%s: DEC out of range -90 to +90\n", sbrtnname);
	exit (-1);
      }
      v1 *= DEG2RAD;
      for (j = 0; j < nflds; j++) pixr[j].dec = v1;
      break;

    case 'Q':			/* Dump Bin -> Ascii */
      for (j = 0; j < nflds; j++) pixr[j].gzflg |= GZF_FTYPE_DMP;
      break;

    case 'S':			/* RA Width */
      v1 = atof(optarg);
      if (v1 < 0 || v1 > width_max) {
	fprintf (stderr, "%s: RA Width range is 0 to %.1lf deg\n",
		 sbrtnname, width_max);
	exit (-1);
      }
      v1 *= DEG2RAD;
      for (j = 0; j < nflds; j++) pixr[j].dra = v1;
      break;

    case 's':			/* Dec Height */
      v1 = atof(optarg);
      if (v1 < 0 || v1 > height_max) {
	fprintf (stderr, "%s: DEC Height range is 0 to %.1lf deg\n",
		 sbrtnname, height_max);
	exit (-1);
      }
      v1 *= DEG2RAD;
      for (j = 0; j < nflds; j++) pixr[j].ddec = v1;
      break;

    case 'Z':			/* short, medium or long form output */
      /* 
     CB_CATID0 
   | CB_CATID1
   | CB_RADEC  
   | CB_SRADEC 
	   | CB_EPOCH  
   | CB_MU     
	   | CB_MUPROB 
   | CB_SMU
	   | CB_SFIT   
   | CB_FITPTS 
   | CB_FLG
   | CB_MAG    
		   | CB_MFLG   
		   | CB_FLDID 
	   | CB_SG  
		   | CB_SVY 
		   | CB_XYRES  
		   | CB_PLTIDX
		   | CB_XIETA
      */

      if        (!strcmp(optarg, "S")) { /* short */
	for (j = 0; j < nflds; j++) 
	  pixr[j].colbits  = 0 | 
	    (CB_CATID0   | CB_CATID1
	     | CB_RADEC  | CB_SRADEC 
	     | CB_MU     | CB_SMU
	     | CB_FITPTS | CB_FLG
	     | CB_MAG);
      } else if (!strcmp(optarg, "M")) { /* medium */
	for (j = 0; j < nflds; j++) 
	  pixr[j].colbits  = 0 | 
	    (CB_CATID0   | CB_CATID1
	     | CB_RADEC  | CB_SRADEC 
	     | CB_EPOCH
	     | CB_MU     | CB_MUPROB | CB_SMU
	     | CB_SFIT   | CB_FITPTS | CB_FLG
	     | CB_MAG    | CB_SG     | CB_XIETA );
      } else if (!strcmp(optarg, "L")) { /* long */
	for (j = 0; j < nflds; j++) 
	  pixr[j].colbits  = 0 | 
	    (CB_CATID0   | CB_CATID1
	     | CB_RADEC  | CB_SRADEC 
	     | CB_EPOCH
	     | CB_MU     | CB_MUPROB | CB_SMU
	     | CB_SFIT   | CB_FITPTS | CB_FLG
	     | CB_MAG    | CB_MFLG   | CB_FLDID
	     | CB_SG     | CB_SVY    | CB_XYRES
	     | CB_PLTIDX | CB_XIETA  );
      } else {
	fprintf (stderr, "%s: Allowed Output Formats are\n", sbrtnname);
	fprintf (stderr, 
		 "   S = short, M = medium, L = long (all fields)\n");
	exit ( -1 );
      }

      break;

    case 'z':			/* output file sort key */
      if        (!strcmp(optarg, "N")) { /* null sort */
	for (j = 0; j < nflds; j++) pixr[j].gzflg += GZF_SSORT_VAL(0);
      } else if (!strcmp(optarg, "R")) { /* sort on RA */
	for (j = 0; j < nflds; j++) pixr[j].gzflg += GZF_SSORT_VAL(SB_RA);
      } else if (!strcmp(optarg, "D")) { /* sort on DEC */
	for (j = 0; j < nflds; j++) pixr[j].gzflg += GZF_SSORT_VAL(SB_DEC);
      } else if (!strcmp(optarg, "M")) { /* sort on MAG */
	for (j = 0; j < nflds; j++) pixr[j].gzflg += GZF_SSORT_VAL(SB_MAG);
      } else if (!strcmp(optarg, "C")) { /* sort on Color */
	for (j = 0; j < nflds; j++) pixr[j].gzflg += GZF_SSORT_VAL(SB_CLR);
      } else if (!strcmp(optarg, "L")) { /* sort on DIST2CTR  */
	for (j = 0; j < nflds; j++) {
	  pixr[j].gzflg += GZF_SSORT_VAL(SB_RADIUS);
	  pixr[j].colbits |= CB_RADIUS;
	}
	
      } else {
	fprintf (stderr, "%s: Allowed Sort Keys are\n", sbrtnname);
	fprintf (stderr, "     N = No sort\n");
	fprintf (stderr, "     R = Sort on RA\n");
	fprintf (stderr, "     D = Sort on DEC\n");
	fprintf (stderr, "     M = Sort on Magnitude\n");
	fprintf (stderr, "     C = Sort on Color\n");
	fprintf (stderr, "     L = Sort on Distance to Center\n");
	exit ( -1 );
      }
      break;

    case 'H':
    case 'h':
    default:			/* default */
      ubc_usage2();

      exit (-2);

      break;
    }
  }

  for (j = 0; j < nflds; j++) {
    if (pixr[j].brtmag > pixr[j].fntmag) {
      fprintf (stderr, 
	    "%s: Value for Bright Magnitude must be < Faint Magnitude\n", 
	       sbrtnname);
      exit ( -1 );
    }

    if (pixr[j].bmrmin > pixr[j].bmrmax) {
      fprintf (stderr, "%s: Value for Color needs Min < Max\n", 
	       sbrtnname);
      exit ( -1 );
    }

    if (pixr[j].mumin > pixr[j].mumax) {
      fprintf (stderr, "%s: Value for Mu needs Min < Max\n", sbrtnname);
      exit ( -1 );
    }

    if (pixr[j].sposmin > pixr[j].sposmax) {
      fprintf (stderr, "%s: Value for Pos'n Error needs Min < Max\n", 
	       sbrtnname);
      exit ( -1 );
    }

    if (pixr[j].smagmin > pixr[j].smagmax) {
      fprintf (stderr, "%s: Value for Magnitude Error needs Min < Max\n", 
	       sbrtnname);
      exit ( -1 );
    }

    if (pixr[j].densmin > pixr[j].densmax) {
      fprintf (stderr, "%s: Value for Density needs Min < Max\n", 
	       sbrtnname);
      exit ( -1 );
    }

    if (pixr[j].minsg > pixr[j].maxsg) {
      fprintf (stderr, "%s: Value for Star/Galaxy needs Min < Max\n", 
	       sbrtnname);
      exit ( -1 );
    }

  /* if circular extraction, make ddec = dra */
    if ( pixr[j].catal & CAT3_EXT_2 ) {
      pixr[j].dra *= 2.0;
      pixr[j].ddec = pixr[j].dra; 
    }

  /* turn off xi,eta output for Dec Band and All Sky */
    if ( pixr[j].catal & CAT3_EXT_3 || !(pixr[j].catal & CAT3_EXT) ) {
      pixr[j].colbits = pixr[j].colbits & (~CB_XIETA);
    }

  /* if rectangle, or box, width_max = height_max */
    if ( (pixr[j].catal & CAT3_EXT_1) || (pixr[j].catal & CAT3_EXT_2) ) {
      if (pixr[j].dra > height_max) {
	fprintf (stderr, 
	  "%s: dRA greater than limit of %lf for Rect. or Circle\n", 
	  sbrtnname, height_max);
	exit ( -1 );
      }
    }

    pixr[j].ra2k = pixr[j].ra;
    pixr[j].dec2k = pixr[j].dec;
  }

  for (j = 0; j < nflds; j++) {
    fprintf (stderr,
	 "%s %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %s %s %d %d %d %d %d %d %lf %s\n",
	     pixr[j].fldnam, 
	     pixr[j].ra2k,    pixr[j].dec2k, 
	     pixr[j].ra,      pixr[j].dec, 
	     pixr[j].dra,     pixr[j].ddec,
	     pixr[j].brtmag,  pixr[j].fntmag, 
	     pixr[j].bmrmin,  pixr[j].bmrmax, 
	     pixr[j].clr,     pixr[j].pname,
	     pixr[j].ori,     pixr[j].gzflg, 
	     pixr[j].ticks,   pixr[j].catal, 
	     pixr[j].surveys, pixr[j].devtype, 
	     pixr[j].cmark,   pixr[j].email);

    fprintf (stderr, 
    "  %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %d %d %lf %d\n",
	     pixr[j].equin,   pixr[j].epoch,   
	     pixr[j].mumin,   pixr[j].mumax,
	     pixr[j].sposmin, pixr[j].sposmax, 
	     pixr[j].smagmin, pixr[j].smagmax, 
	     pixr[j].densmin, pixr[j].densmax,
	     pixr[j].minsep,  pixr[j].msdmag,
	     pixr[j].minsg,   pixr[j].maxsg,
	     pixr[j].pixscale,pixr[j].colbits);

  }

  return ( pixr );
}

/* TITLE: ubc_usage2 - usage of usno-b extraction */
int ubc_usage2 ()
{
  fprintf (stderr, 
    "usage: ubcone - extract a region from the USNO-B1.0 Catalogue\n");
  fprintf (stderr, " Option Description                      Default\n");
  fprintf (stderr, " ------ -----------                      -------\n");
  fprintf (stderr, " -B     Field Name                       \"Field\"\n");
  fprintf (stderr, " -F     RegionType (R, B, C)                   R\n");
  fprintf (stderr, "          (R=Rectangle,C=Circle,B=DecBand)\n");
  fprintf (stderr, "\n");  
  fprintf (stderr, " -h     to get this help output\n");
  fprintf (stderr, " -H     to get this help output\n");
  fprintf (stderr, "\n");  
  fprintf (stderr, " -I     Bright Magnitude                     0.0\n");
  fprintf (stderr, " -i     Faint Magnitude                    100.0\n");
  fprintf (stderr, " -L     Primary Sort Color (B1 R1 B2 R2 I2)   R2\n");
  fprintf (stderr, " -O     Output File                        \"Tmp\"\n");
  fprintf (stderr, " -P     RA(hrs)                              0.0\n");
  fprintf (stderr, " -p     DEC(deg)                             0.0\n");
  fprintf (stderr, " -Q     Turn on Binary -> Ascii Dump Mode\n");
  fprintf (stderr, " -S     RA Width/Circle Radius(deg)          0.166667\n");
  fprintf (stderr, " -s     Dec Height(deg)                      0.166667\n");
  fprintf (stderr, " -Z     Output Format (S M L)                  M\n");
  fprintf (stderr, "          (S=Small, M=Medium, L=all fields)\n");
  fprintf (stderr, " -z     output file sort key (N R D M C L)     N\n");
  fprintf (stderr, "          (Sort in N=none, R=RA, D=Dec, M=Mag, C=Color, L=Dist. to Center)\n");

  fprintf (stderr, "\nMax Subset Size: %d\n", MAXUBSTARS);
  fprintf (stderr, "Max Box Size is: %lf x %lf [deg]\n", UBC_MAXBOX, UBC_MAXBND);
  fprintf (stderr, "\nVersion: %s\n", UBVER);

  return ( 0 );
}
