/* Written by Stephen Levine @ the US Naval Observatory, Flagstaff */
/* Copyright USNO, all rights reserved */

#include "./ubsquare.h"

/* TITLE: ubc_eatit */
int ubc_eatit(double r1, double r2, int dz, double d1, double d2, 
	     PIXREGION *pixr, STARCAT *sc)
{
  char *sbrtnname={"ubc_eatit"};
/*
...Ingest ACC And Loop Over DAT
*/
  int i, i1, i2, fr, nr, c_roopen_(), fd, nlb, err, cpnb;
  int c_position_(), n, nmost, nlast, ir1, ir2, id1, id2;
  int c_closer_(), c_reader_(), c_rawreader_(), m, j;
  char lb[64], lba[8];
  FILE *fp;

/*
...Eat The ACC File
*/
  if (dz != ub_oldzone) { 
    sprintf (lba, "%4.4d", dz);
    sprintf (lb, "%s/%3.3s/b%4.4d.acc", b_base, lba, dz);
    if ( (fp = fopen ( lb, "r")) == (FILE *)NULL ) {
      fprintf (stderr, "%s: unable to open acc file %s\n", sbrtnname, lb);
      return ( -1 );
    }
    for (i=1; i <= NACC; i++) {
      fscanf (fp, "%*lf %d %d", &(ub_frec[i]),&(ub_nrec[i]));

    }
    fclose (fp);
    ub_oldzone = dz;
  }
#ifdef DEBUG
  fprintf (stderr, "%s: %lf %lf %d %lf %lf\n", sbrtnname, r1, r2, dz, d1, d2);
#endif

/*
...Compute Offset And Length
*/
  i1 = r1/3.75;
  i1 = MAX(1,MIN(NACC,i1+1));
  i2 = r2/3.75;
  i2 = MAX(1,MIN(NACC,i2+1));
  fr = ub_frec[i1]-1;
  nr = 0;
  for (i=i1; i <= i2; i++) {
    nr = nr+ub_nrec[i];
  }

/*
...Open And Position File
*/
  sprintf (lba, "%4.4d", dz);
  sprintf (lb, "%s/%3.3s/b%4.4d.cat", b_base, lba, dz);
  fd = c_roopen_(lb);
  if (fd < 3) {
    fprintf (stderr, "%s: Cannot Open %s\n",sbrtnname, lb);
    return ( -2 );
  }

  if (fr > 0) {			/* move in # of bytes if fr>0 */
    cpnb = NINTREC*4*fr;
    err = c_position_(&fd,&cpnb);
    if (err <= 0) {
      fprintf (stderr, "%s: Almost Fatal Error Accessing %s\n", sbrtnname, lb);
      return ( -3 );
    }
  }

/*
...Set Up Search Parameters
*/
  n = (int)((nr-1)/NCHUNK) + 1;
  if (n > 1) {
    nmost = NCHUNK;
    nlast = nr - (n-1)*NCHUNK;
  } else {
    nmost = 0;
    nlast = nr;
  }
  ir1 = CONVERT*r1;
  ir2 = CONVERT*r2;
  id1 = CONVERT*(d1+90.0);
  id2 = CONVERT*(d2+90.0);
#ifdef DEBUG
  fprintf (stderr, "%s: Z= %4d, RA(%9d:%9d) SPD(%9d:%9d)\n",
	   sbrtnname, ub_oldzone,ir1,ir2,id1,id2);
#endif

/* ONLY if we are in Dump mode break out if dec lims are the same */
  if ( pixr->gzflg & GZF_FTYPE_DMP ) {
    if ( ABS(id1 - id2) <= 1 ) {
      fprintf (stderr, "%s: ABS(id1(%d) - id2(%d)) <= 1 skipping\n", sbrtnname,
               id1, id2);
      return ( 0 );
    }
  }

/*
...Do The Search
*/

#ifdef DEBUG
  fprintf (stderr, "%s: Nchunks = %d NRecs = %d \n", sbrtnname, n, nr);
#endif

  for (i=1; i <= n; i++) {
    if (i == n) m = nlast;
    else        m = nmost;
    cpnb = NINTREC*4*m;
    err = c_rawreader_(&fd,ub_buf,&cpnb);
    if (err != 0) {
      fprintf (stderr, "%s: Almost Fatal Error Accessing %s\n",sbrtnname, lb);
      return ( -3 );
    }
    for (j=0; j < m; j++) {
      if (ub_buf[j][0] >= ir1) {
	if (ub_buf[j][0] <= ir2) {
	  if ((ub_buf[j][1] >= id1) && (ub_buf[j][1] <= id2)) {
	    ub_nsav++;
	    ubc_saveit(j, dz, fr+1+j, pixr, sc);
	  }
	} else {
	  err = c_closer_(&fd);
	  return ( 0 );
	}
      }
    }
    fr = fr+m;
  }

/*
...All Done
*/

  err = c_closer_(&fd);

  return ( 0 );
}
