/* Written by Stephen Levine @ the US Naval Observatory, Flagstaff */
/* Copyright USNO, all rights reserved */

#include "./ubsquare.h"

/* TITLE: ub_xyrcrnr - */
/*
c
c r,d = ra dec of center, sr = ra half width, sd = dec half width
c...Get Corners On The Basis Of (Xi,Eta) And Compute Search Parameters
c
c from xycorner of D.G. Monet
c modified by S. Levine July 1997 to handle rectangular region
c
c	WARNING - Uses SLALIB
c
*/

int ub_xyrcrnr(double r, double d, double sr, double sd, int catal,
	    double zone_wid)
{
  char *sbrtnname={"ub_xyrcrnr"};
  double radr, radd, radsr, radsd, dmin, dmax, zmin, zmax;
  double rcr[4], dcr[2];
  int    i, j, z1, z2, izone_w;

  void rdmnmx(double, double, double, double, int *, double *, double *);
  void dbmnmx(double, double, double, double, int *, double *, double *);

#ifdef DEBUG
  fprintf (stderr, "%s: input - %lf %lf %lf %lf\n", sbrtnname, r, d, sr, sd);
#endif

/*
...Sanity Checks
*/
  if ((r < 0.0)||(r >= TWOPI)) {
    fprintf (stderr, "%s: Illegal Input: R= %10.6lf  D= %10.6lf\n", 
	     sbrtnname, r, d);
    leave( -1 );
  }
  if ((d < -HALFPI)||( d > HALFPI)) {
    fprintf (stderr, "%s: Illegal Input: R= %10.6lf  D= %10.6lf\n", 
	     sbrtnname, r, d);
    leave( -2 );
  }
  if ((sr <= 0.0)||(sd <= 0.0)) {
    fprintf (stderr, "%s: Illegal Size= %10.6lf %10.6lf\n",
	     sbrtnname, sr, sd);
    leave( -3 );
  }

/*
...Use SLALIB To Get Corners
*/
  radr  = r;
  radd  = d;
  radsr = sr;
  radsd = sd;

  if ( catal == 1 || catal == 2 )               /* 1 = rectangle, 2 = circle */
    rdmnmx (radr, radd, radsr, radsd, &ub_nra, rcr, dcr);
  else if ( catal == 3 )	                /* 3 = dec band */
    dbmnmx (radr, radd, radsr, radsd, &ub_nra, rcr, dcr);
  else if ( catal == 0 )	                /* 0 = whole sky */
    wsmnmx (radr, radd, radsr, radsd, &ub_nra, rcr, dcr);
  else
    rdmnmx (radr, radd, radsr, radsd, &ub_nra, rcr, dcr);

  ub_rfrst[1] = rcr[0] * 15.0;  ub_rlast[1] = rcr[1] * 15.0;
  ub_rfrst[2] = rcr[2] * 15.0;  ub_rlast[2] = rcr[3] * 15.0;
  dmin     = dcr[0];         dmax     = dcr[1];

/*
...Compute Dec Corners And Zones
*/

  /*  zone_wid = 7.5;	*/	/* width of each dec zone in deg */
  izone_w  = (int)(zone_wid * 10.0); /* 10 x integer width */

#ifdef DEBUG
  fprintf (stderr, "%s: RA lims  = %8.4lf %8.4lf %8.4lf %8.4lf\n", 
	   sbrtnname, ub_rfrst[1], ub_rlast[1], ub_rfrst[2], ub_rlast[2]);
  fprintf (stderr, "%s: DEC lims = %8.4lf %8.4lf\n", sbrtnname, dmin, dmax);
#endif
  z1 = (dmin+90.0)/zone_wid;
  z2 = (dmax+90.0)/zone_wid;
  ub_ndec = z2+1-z1;
  j = 0;
  for (i = z1; i <= z2; i++) {
    zmin = (double)i*zone_wid - 90.0;
    zmax = zmin+zone_wid;
    j++;
    ub_dzone[j] = i*izone_w;
    ub_dfrst[j] = MAX(dmin,zmin);
    ub_dlast[j] = MIN(dmax,zmax);
#ifdef SKIP
    fprintf (stderr, "%s: %d %8.4lf %8.4lf\n", 
	     sbrtnname, ub_dzone[j], ub_dfrst[j], ub_dlast[j]);
#endif
  }
  return ( 0 );
}
