/* Written by Stephen Levine @ the US Naval Observatory, Flagstaff */
/* Copyright USNO, all rights reserved */

#define SLOGLINE   (1024)
#define SVYLINE     (256)
#define SVYLBL       (16)

/* TypeDefs -------------------------------------------------- */

typedef struct pixregion {	/* Pixel Region Info */
  char   fldnam[SVYLINE];	 /* field name */
  double ra2k, dec2k;		 /* Center Ra & Dec (J2000.0) (radians) */
  double ra, dec;		 /* Center Ra & Dec (Equinox) (radians) */
  double dra, ddec;		 /* width in Ra & Dec (radians) */
  double equin, epoch;		 /* request equinox, epoch */
  double brtmag, fntmag;	 /* bright and faint magnitudes (mag) */
  double bmrmin, bmrmax;	 /* Color range: B-R min and max (mag) */
  char   clr[SVYLBL];		 /* color - for primary magnitude */
  char   clr0m1[2][SVYLBL];	 /* color - for mag_0 minus mag_1 */
  char   pname[SVYLINE];	 /* tmp root file name */
  int    ori, gzflg, ticks;	 /* orientation, gzip flag, tick marks */
  int    catal, surveys;	 /* catalogues, image surveys */
  int    devtype;		 /* output dev. type */
  double cmark;			 /* center mark */
  char   email[SVYLINE];	 /* e-mail address for notice */

  double pixscale;		 /* arcsec/pix or /mm for x,y cat. output */
  double mumin,   mumax;	 /* min and max proper motion (mas/yr) */
  double sposmin, sposmax;	 /* min and max position error (mas) */
  double smumin,  smumax;	 /* min and max proper motion error (mas/yr) */
  double smagmin, smagmax;	 /* min and max photometric error (mag) */
  double densmin, densmax;	 /* min and max density (#/sq. deg) */
  double minsep,  msdmag;	 /* min separation between objects (radians) */
				 /* msdmag = # mags for exclusion */

  int    minsg,   maxsg;	 /* min and max star/galaxy index (0-11) */

  int    colbits;		 /* bits for desired catalogue columns */

} PIXREGION;

typedef struct gencatal {	/* generic catalogue info */
  int    idnum;			 /* increasing index number */
  int    flg;			 /* flag bits for this survey (for incl/exl) */
  char   id[SVYLINE];		 /* survey name */
  int    nflds;			 /* number of fields */
  int    platesperfld;		 /* number of plates per field */
  int    addplates;		 /* number of additional plates */
  int    totalplate;		 /* total number of plates */
  double halfwidth;		 /* plate half width in degrees */
  double platescale;		 /* plate scale in arcsec/mm */
  double pixsize;		 /* pixelsize, arcsec/pix */
  int    ccdx, ccdy;		 /* pixel dimensions of ccd */
  int    fpx, fpy;		 /* # of footprints per plate in x, y dimens */
  char   gcat[SVYLINE];		 /* name of generic catalogue file */
  char   orgcat[SVYLINE];	 /* name of original catalogue file */
  int    scanori;		 /* orientation of plates when scanned */
				  /* possible orientations
				     0 = (N up E right)  
				     1 = (N up E left)   
				     2 = (N down E right)
				     3 = (N down E left) 
				     4 = (N right E up)  
				     5 = (N left E up)   
				     6 = (N right E down)
				     7 = (N left E down) 
				  */ 
} GENCATAL;

typedef struct svyinf {		/* basic survey & plate info */
  char   id[SVYLINE];		   /* ID - survey code or field name/# */
  char   label[SVYLINE];	   /* label - survey descrip. or plate # */
  double mnra, mxra, mndec, mxdec; /* min, max Ra & Dec */
  int    nx, ny;		   /* # of FP/plates in x and y dir */
  int    nimgs;			   /* number of footprints on plates */
				   /* or plates in survey (= nx*ny) */
  int    flgs;			   /* flag bits */
} SVYINF;

typedef struct dltinf {		/* information about DLT's */
  char   id[SVYLBL];		   /* VolID of the DLT */
  char   plid[SVYLBL];		   /* plate id on tape */
  int    num;			   /* plate number on DLT */
  int    nfps;			   /* number of footprints on plate */
  int    nfile, nfppf;		   /* # of archive files, # of FP/file */
  int    preskip;		   /* # of real files to skip to get to */
				   /* the beginning of this plate */
} DLTINF;

typedef struct platelog {	/* information about a specific plate */
  int    id;			   /* ID - survey code or field name/# */
  char   svynam[SVYLINE];	   /* name of survey that contains this img */
  char   label[SVYLINE];	   /* label - survey descrip. or plate # */
  char   dbid[SVYLBL];		   /* database plate id 6 char (up to SVYLBL)*/
  double date,  time;		   /* date and time of exposure */
  double rac50, decc50;		   /* plate center Ra & Dec B1950 */
  double rac2K, decc2K;		   /* plate center Ra & Dec J2000/ICRS */
  double scale;			   /* plate scale on this plate [as/pix] */
  double micronpix;		   /* microns per pixel */
  int    ccdx, ccdy;		   /* pixel dimensions of ccd image */
  double mnra, mxra, mndec, mxdec; /* min, max Ra & Dec */
  double platenx, plateny;         /* x, y position of platen for image (mm)*/
  double crra[4], crdec[4];	   /* corner ra and dec */
  double crx[4],  cry[4];	   /* corner x and y */
  char   emul[SVYLBL], filt[SVYLBL];   /* emulsion type and filter */
  char   expos[SVYLBL], hrang[SVYLBL]; /* exposure duration, Hour Angle */
  int    scanori;		   /* orientation of plates when scanned */
  int    nimgs, *fimgs, notondisk; /* number of, and id of footprints wanted */
				   /*  and if it is already on disk (1) */
  double *fpx, *fpy, fangle;	   /* footprint x and y pos'n for ra/dec */
				   /*  and angle of rotation to line up fp */
  DLTINF dlt;			   /* DLT information */
} PLATELOG;

typedef struct surveylog {	/* information  about multiple plates */
  SVYINF    sv;			/* survey name and info */
  PLATELOG *pl;			/* info for each plate */
} SURVEYLOG;

typedef struct dltlib {
  int nentries;			/* number of entries in DLT file */
  DLTINF *dlt;
} DLTLIB;

typedef struct starcat {	/* Star Catalogue Object */
  int    colbits;		/*  bits for which columns exist in this */
  char   catname[SVYLINE];		/*  catalogue name */
  int    nstars;		/*  number of stars in this list */
  int    nmagcol;		/*  number of mag. columns */
  char   magid[8][8];		/*  names of the magnitudes, 8 up to 8 char */
  int    *catid[2];		/*  catalogue id number */
  double *ra, *dec;		/*  catalogue ra, dec */
  double *sra, *sdec;		/*  sigma catalogue ra, dec */
  double equinox, *epoch;	/*  catalogue equinox, object epoch */
  int    *mura, *mudec, *muprob;/*  catalogue prop. motion, + probability */
  int    *smura, *smudec;	/*  sigma catalogue prop. motion */
  int    *srafit, *sdecfit;	/*  sigma ra/dec fit */
  int    *fitpts;		/*  # of pts in fit */
  int    *err;			/*  error field */
  char   *flg;			/*  one byte for flags */
  double **mag, **smag;		/*  catalogue magnitudes (up to 8) */
  int    **mflg;		/*  magnitude related flags */
  int    **fldid;		/*  plate/field id */
  int    **sg;			/*  star/galaxy classification */
  double **xres, **yres;	/*  x,y residuals in fit by plate */
  int    **pltidx;		/*  index into plate files */

  double *el, *eb;		/*  computed ecliptic coords (lam,beta) */
  double *gl, *gb;		/*  computed galactic coords (l,b) */
  double *xi, *eta;		/*  xi,eta coords for star on focal plane */
  double *x,  *y;		/*  x,y coords for star in ROI */
  double *xm, *ym;		/*  ``measured'' x,y coords for star in ROI */
  double *estmag;		/*  estimated magnitude */
} STARCAT;

typedef struct opobjs {		/* OverPlot Objects */
  char   id[SVYLINE];		/*  ID label */
  int    nobjs;			/*  number of objects in this list */
  int    connect;		/*  connect flag */
				/*   0 = pts only */
				/*   1 = pts + labels */
				/*   2 = connected pts */
				/*   3 = connected pts + labels */
  int    ptyp;			/*  basic point style */
  double *ra;			/*  obj ra */
  double *dec;			/*  obj dec */
  double *mag;			/*  obj  magnitude */
  char   **plbl;		/*  obj label */
  int    *pclr;			/*  obj plot color */
				/*   0 = "default" */
				/*   1 = "white"   */
				/*   2 = "black"   */
				/*   3 = "red"     */
				/*   4 = "green"   */
				/*   5 = "blue"    */
				/*   6 = "cyan"    */
				/*   7 = "magenta" */
				/*   8 = "yellow"  */
  int    *psymb;		/*  obj point type */
				/*   0 = PNT (point) */
				/*   1 = CIR (circle) */
				/*   2 = SQR (square) */
				/*   3 = REC (rectangle) */
				/*   4 = ELL (ellipse) */
				/*  CIRCLE SQR/RECT   ELLIPSE */
  double *a;			/*  radius,width,     semi-maj. axis */
  double *ba;			/*         height,    axis ratio */
  double *pa;			/*                    position angle */
} OPOBJS;
