/* Written by Stephen Levine @ the US Naval Observatory, Flagstaff */
/* Copyright USNO, all rights reserved */

#ifdef STANDALONE
#include <stdio.h>
#include "./tdefs.h"
#include "./flagbits.h"
#include "./Sla_bit/slamac.h"
#else
#include "../Incl/hdrs.h"
#endif

/* TITLE: srchinfo - write search info at top of file */
int srchinfo (FILE *fp, char *catid, PIXREGION *pixr, int npts)
{
  char sv1[80];

  sprintf (sv1,"'%s'", pixr->fldnam);
  wfstr (fp, "FIELD", sv1, " ", "Field Name");

  sprintf (sv1,"'%s'", catid);
  wfstr (fp, "CAT_ID", sv1, " ", "Catalogue ID");

  wfint (fp, "NOBJECTS",  npts, " ", "Number of Objects In File");

  sprintf (sv1,"%c%6.1lf %7.2lf", 
	   ((pixr->equin == 1950.0) ? 'B' : 'J'), pixr->equin, pixr->epoch);
  wfstr (fp, "EQUINOX", sv1," ", "Equinox, Epoch");

  wfdbl (fp, "RA",  pixr->ra*DR2D/15.0, " ", "Center RA  [hrs]");
  wfdbl (fp, "DEC", pixr->dec*DR2D,     " ", "Center DEC [deg]");


  if (pixr->catal & CAT3_EXT_1) {	        /* rectangular extraction */
    wfdbl (fp, "DRA",  pixr->dra  * DR2D, " ", "Box Width  [deg]");
    wfdbl (fp, "DDEC", pixr->ddec * DR2D, " ", "Box Height [deg]");

  } else if (pixr->catal & CAT3_EXT_2) {	/* circular extraction */
    wfdbl (fp, "RADIUS", 0.5*pixr->dra*DR2D, " ", "Circle Radius [deg]");

  } else if (pixr->catal & CAT3_EXT_3) {	/* dec band extraction */
    wfdbl (fp, "DRA",  24.0       * 15.0, " ", "RA Width [deg]");
    wfdbl (fp, "DDEC", pixr->ddec * DR2D, " ", "Dec Band Height [deg]");

  } else if (!(pixr->catal & CAT3_EXT)) {	/* all sky extraction */
    wfdbl (fp, "DRA",  24.0       * 15.0, " ", "RA Width [deg]");
    wfdbl (fp, "DDEC", 180.0, " ", "Dec Band Height [deg]");

  } else {
    wfdbl (fp, "DRA",  pixr->dra  * DR2D, " ", "Box Width  [deg]");
    wfdbl (fp, "DDEC", pixr->ddec * DR2D, " ", "Box Height [deg]");
  }

  sprintf (sv1,"'%s'", pixr->clr);
  wfstr  (fp, "MAGNITUD",  sv1,       " ", "Primary Sort Color");
  wfdbl2 (fp, "BR_MAG", pixr->brtmag, " ", "Bright Magnitude Limit [mag]");
  wfdbl2 (fp, "FT_MAG", pixr->fntmag, " ", "Faint Magnitude Limit [mag]");

  sprintf (sv1,"'%s-%s'", pixr->clr0m1[0], pixr->clr0m1[1]);
  wfstr  (fp, "COLOR",  sv1,       " ", "Color0 minus Color1 (C0-C1)");
  wfdbl2 (fp, "BR_MIN", pixr->bmrmin,    " ", "Color Limit: Min. C0-C1 [mag]");
  wfdbl2 (fp, "BR_MAX", pixr->bmrmax,    " ", "Color Limit: Max. C0-C1 [mag]");

  wfdbl2 (fp, "MU_MIN", pixr->mumin, " ", 
	  "Min. Proper Motion [mas/yr]");
  wfdbl2 (fp, "MU_MAX", pixr->mumax, " ", 
	  "Max. Proper Motion [mas/yr]");

  wfdbl2 (fp, "SPOS_MIN", pixr->sposmin, " ", 
	  "Min. Position Error [mas]");
  wfdbl2 (fp, "SPOS_MAX", pixr->sposmax, " ", 
	  "Max. Position Error [mas]");

  wfdbl2 (fp, "SMU_MIN", pixr->smumin, " ", 
	  "Min. Proper Motion Error [mas/yr]");
  wfdbl2 (fp, "SMU_MAX", pixr->smumax, " ", 
	  "Max. Proper Motion Error [mas/yr]");

  wfdbl2 (fp, "SD_MIN", pixr->densmin, " ", 
	  "Minimum Surface Density [N/deg^2]");
  wfdbl2 (fp, "SD_MAX", pixr->densmax, " ", 
	  "Maximum Surface Density [N/deg^2]");
/*
  wfdbl  (fp, "SEP_MIN",pixr->minsep*DR2D, " ", 
	  "Minimum Separation [deg]");
*/
  sprintf (sv1,"%.6lf %.2lf", pixr->minsep*DR2D, pixr->msdmag);
  wfstr (fp, "SEP_MIN", sv1," ", "Minimum Separation [deg] & [mag]");


  wfend (fp);

  return ( 0 );
}

int wfstr (FILE *fp, char *kwrd, char *val1, char *val2, char *comm)
{
  fprintf (fp, "%-8s= %-20s%9s/%-40s\n", kwrd, val1, val2, comm);
  return ( 0 );
}

int wfint (FILE *fp, char *kwrd, int val1, char *val2, char *comm)
{
  fprintf (fp, "%-8s= %-20d%9s/%-40s\n", kwrd, val1, val2, comm);
  return ( 0 );
}

int wfdbl (FILE *fp, char *kwrd, double val1, char *val2, char *comm)
{
  fprintf (fp, "%-8s= %-20lf%9s/%-40s\n", kwrd, val1, val2, comm);
  return ( 0 );
}

int wfdbl2 (FILE *fp, char *kwrd, double val1, char *val2, char *comm)
{
  fprintf (fp, "%-8s= %-20.2lf%9s/%-40s\n", kwrd, val1, val2, comm);
  return ( 0 );
}

int wfend (FILE *fp)
{
  fprintf (fp, "%-3s%-77s\n","END", " ");
  return ( 0 );
}
