/* Written by Stephen Levine @ the US Naval Observatory, Flagstaff */
/* Copyright USNO, all rights reserved */

#include <math.h>

/* MACROS */
#define	ABS(x)		(((x) < 0) ? -(x) : (x))
#define	MAX(a,b)	(((a) < (b)) ? (b) : (a))
#define	MIN(a,b)	(((a) > (b)) ? (b) : (a))

#define PI	 3.141592653589793238462643e0
#define TWOPI    6.283185307179586476925286766559e0
#define DEG2RAD  (PI / 180.0)

/* TITLE: rdmnmx */
/* given ra, dec of center, and ra and dec full widths  */
/* return min and max ra and decs so that region covers */
/* xi, eta rectangle of desired widths */
/* return values: nz = # of regions - to handle crossing 0/24 hr line */
/*  rcr[4] = ramin1, ramax1, ramin2, ramax2 */
/*  dcr[2] = decmin, decmax */

void rdmnmx (double rac,  double decc, double ra_w, double dec_w,
	     int *nz, double *rcr, double *dcr)
{
  double r1, r2, r3, r4, r5, r6, rmin, rmax;
  double d1, d2, d3, d4, dmin, dmax;
  double racr, deccr, rawr, decwr;
  int    i;
  void sdtp2s (double, double, double, double, double *, double *);
  
  racr   = rac;
  deccr  = decc;
  rawr   = 0.5 * ra_w;
  decwr  = 0.5 * dec_w;

  for (i = 0; i < 4; i++){ rcr[i] = 0.0; }
  for (i = 0; i < 2; i++){ dcr[i] = 0.0; }

  sdtp2s(-rawr, -decwr, racr, deccr, &r1, &d1);
  sdtp2s( rawr, -decwr, racr, deccr, &r2, &d2);
  sdtp2s(-rawr,  decwr, racr, deccr, &r3, &d3);
  sdtp2s( rawr,  decwr, racr, deccr, &r4, &d4);
  sdtp2s( 0.0 , -decwr, racr, deccr, &r5, &dmin);
  sdtp2s( 0.0 ,  decwr, racr, deccr, &r6, &dmax);

/* Handle RA */
  rmin = MIN(r1, MIN(r2, MIN(r3, r4)));
  rmax = MAX(r1, MAX(r2, MAX(r3, r4)));

  if ((rmax - rmin) < PI) {
    *nz = 1; rcr[0] = rmin / DEG2RAD / 15.0; rcr[1] = rmax / DEG2RAD / 15.0;
  } else {
    rmin = TWOPI;    rmax = 0.0;

    if (r1 <= PI) rmax = MAX(rmax,r1); 
    else          rmin = MIN(rmin,r1); 

    if (r2 <= PI) rmax = MAX(rmax,r2);
    else          rmin = MIN(rmin,r2);

    if (r3 <= PI) rmax = MAX(rmax,r3);
    else          rmin = MIN(rmin,r3);
    
    if (r4 <= PI) rmax = MAX(rmax,r4);
    else          rmin = MIN(rmin,r4);

    *nz = 2; 
    rcr[0] = 0.0;                   rcr[1] = rmax / DEG2RAD / 15.0;
    rcr[2] = rmin / DEG2RAD / 15.0; rcr[3] = 24.0;
  }

/* Handle DEC */
  if ((decc - decwr) < -0.5 * PI) {
    dmin = -89.999999 * DEG2RAD;
    *nz = 1;
    rcr[0] = 0.0; rcr[1] = 24.0;
  }
  if ((decc + decwr) > 0.5 * PI) {
    dmax = 89.999999 * DEG2RAD;
    *nz = 1;
    rcr[0] = 0.0; rcr[1] = 24.0;
  }

  dcr[0] = MIN(d1,MIN(d2,dmin)) / DEG2RAD;
  dcr[1] = MAX(d3,MAX(d4,dmax)) / DEG2RAD;
}

/* TITLE: dbmnmx */
/* for a band in declination, with a given ra, dec center */
/* and width in dec, */
/* return min and max ra and decs so that region covers */
/* xi, eta rectangle of desired widths */
/* return values: nz = # of regions - to handle crossing 0/24 hr line */
/*  rcr[4] = ramin1, ramax1, ramin2, ramax2 */
/*  dcr[2] = decmin, decmax */

void dbmnmx (double rac,  double decc, double ra_w, double dec_w,
	     int *nz, double *rcr, double *dcr)
{
  double dmin, dmax;
  int    i;
  
  for (i = 0; i < 4; i++){ rcr[i] = 0.0; }
  for (i = 0; i < 2; i++){ dcr[i] = 0.0; }

/* Set RA limits */
  *nz = 1; rcr[0] = 0.0; rcr[1] = 24.0;

/* Handle DEC */
  dmin = decc - 0.5 * dec_w;
  dmax = decc + 0.5 * dec_w;

  if ((decc - 0.5 * dec_w) < -0.5 * PI) {
    dmin = -89.999999 * DEG2RAD;
  }
  if ((decc + 0.5 * dec_w) > 0.5 * PI) {
    dmax = 89.999999 * DEG2RAD;
  }

  dcr[0] = dmin / DEG2RAD;
  dcr[1] = dmax / DEG2RAD;
}

/* TITLE: wsmnmx */
/* Whole sky limits in hours and degrees */
/* return min and max ra and decs so that region covers */
/* whole sky (ie, ra = 0,24 hr  & dec = -90,90 deg) */
/* return values: nz = # of regions - to handle crossing 0/24 hr line */
/*  rcr[4] = ramin1, ramax1, ramin2, ramax2 */
/*  dcr[2] = decmin, decmax */

void wsmnmx (double rac,  double decc, double ra_w, double dec_w,
	     int *nz, double *rcr, double *dcr)
{
  int    i;
  
  for (i = 0; i < 4; i++){ rcr[i] = 0.0; }
  for (i = 0; i < 2; i++){ dcr[i] = 0.0; }

/* Set RA limits to 0 and 24 hours, 1 zone */
  *nz = 1; rcr[0] = 0.0; rcr[1] = 24.0;

/* Set DEC limits to -90 and +90 deg */
  dcr[0] = -89.999999;
  dcr[1] = +89.999999;
}

/*
*+
*        -----
* TITLE: DTP2S
*        -----
*
*  Transform tangent plane coordinates into spherical
*  (double precision)
*
*  Given:
*     XI,ETA      dp   tangent plane rectangular coordinates
*     RAZ,DECZ    dp   spherical coordinates of tangent point
*
*  Returned:
*     RA,DEC      dp   spherical coordinates (0-2pi,+/-pi/2)
*
*  Called:        sla_DRANRM
*
*  P.T.Wallace   Starlink   24 July 1995
*
*  Copyright (C) 1995 Rutherford Appleton Laboratory
*- converted from fortran version of slalib 
*/
void sdtp2s (double xi, double eta, 
	     double raz, double decz, 
	     double *ra, double *dec)
{
  double sdecz,cdecz,denom, sdranrm(double);

  sdecz=sin(decz);
  cdecz=cos(decz);

  denom=cdecz-eta*sdecz;

  *ra  = sdranrm(atan2(xi,denom)+raz);
  *dec = atan2(sdecz+eta*cdecz,sqrt(xi*xi+denom*denom));
}

/*
*+
*        ------
* TITLE: DRANRM
*        ------
*
*  Normalize angle into range 0-2 pi  (double precision)
*
*  Given:
*     ANGLE     dp      the angle in radians
*
*  The result is ANGLE expressed in the range 0-2 pi (double
*  precision).
*
*  P.T.Wallace   Starlink   23 November 1995
*
*  Copyright (C) 1995 Rutherford Appleton Laboratory
*- converted from fortran version of slalib 
*/
double sdranrm (double slangle)
{  
  double v1;

  v1 = fmod(slangle,TWOPI);
  if (v1 < 0.0) v1=v1 + TWOPI;

  return (v1);
}

