/* Written by Stephen Levine @ the US Naval Observatory, Flagstaff */
/* Copyright USNO, all rights reserved */

#include <string.h>

#define LUB(x,cr1,cr2)  (((x)>=(cr1)&&(x)<=(cr2)) ? 1 : 0 )
#define LUB2(x,cr1,cr2) (((x)>=(cr1*cr1)&&(x)<=(cr2*cr2)) ? 1 : 0 )

/* TITLE: magclr_match - does object meet magnitude and color criteria? */
/*                    return 1 = satisfies criteria */
/*                           0 = does NOT satisfies criteria */
int magclr_match ( char *clr, double brmag, double ftmag, 
		double bmrmin, double bmrmax,
		double blu, double red )
{
  double v1;

  if ((!strncmp(clr, "B", 1) && LUB(blu,brmag,ftmag)) ||
      (!strncmp(clr, "R", 1)  && LUB(red,brmag,ftmag)) ){
    v1 = blu - red;
    return ( LUB(v1,bmrmin,bmrmax) );
  }

  return ( 0 );
}

/* TITLE: mag_match - does object meet magnitude criterion? */
/*                    return 1 = satisfies criterion */
/*                           0 = does NOT satisfies criterion */
int mag_match ( double magni, double brmag, double ftmag )
{
  return ( LUB(magni,brmag,ftmag) );
}

/* TITLE: clr_match - does object meet color criterion? */
/*                    return 1 = satisfies criterion */
/*                           0 = does NOT satisfies criterion */
int clr_match ( double magni0, double magni1, double clrmin, double clrmax )
{
  return ( LUB(magni0-magni1,clrmin,clrmax) );
}

/* TITLE: mu_match - does object meet proper motion criteria? */
/*                    input mura, mudec in milli-arcsec/yr */
/*                          (ra_dot * cos(dec), dec_dot) */
/*                    return 1 = satisfies criteria */
/*                           0 = does NOT satisfies criteria */
int mu_match ( double mumin, double mumax, double mura, double mudec )
{
  double v1;

  v1 = ( mura * mura + mudec * mudec );
  return ( LUB2(v1,mumin,mumax) );
}

/* TITLE: posn_err_match - does object meet position error criteria? */
/*          ie, MinPosnErr^2 <= sig_ra^2 + sig_dec^2 <= MaxPosnErr^2 */
/*                    input sig_ra, sig_dec in arcsec */
/*                    return 1 = satisfies criteria */
/*                           0 = does NOT satisfies criteria */
int posn_err_match ( double minposnerr, double maxposnerr,
		     double sig_ra, double sig_dec )
{
  double v1;

  v1 = ( sig_ra * sig_ra + sig_dec * sig_dec );
  return ( LUB2(v1,minposnerr,maxposnerr) );
}
      
/* TITLE: mag_err_match - does object meet magnitude error criteria? */
/*          ie, MinMagErr <= sig_mag <= MaxMagErr */
/*                    input sig_mag in magnitudes */
/*                    return 1 = satisfies criteria */
/*                           0 = does NOT satisfies criteria */
int mag_err_match ( double minmagerr, double maxmagerr, double sig_mag )
{
  return ( LUB(sig_mag,minmagerr,maxmagerr) );
}
      
/* TITLE: mu_err_match - does object meet proper motion error criteria? */
/*          ie, MinMuErr^2 <= sig_mura^2 + sig_mudec^2 <= MaxMuErr^2 */
/*                    input sig_mura, sig_mudec in arcsec */
/*                          (ra_dot * cos(dec), dec_dot) */
/*                    return 1 = satisfies criteria */
/*                           0 = does NOT satisfies criteria */
int mu_err_match ( double minmuerr, double maxmuerr, 
		   double sig_mura, double sig_mudec )
{
  double v1;

  v1 = ( sig_mura * sig_mura + sig_mudec * sig_mudec );
  return ( LUB2(v1,minmuerr,maxmuerr) );
}

/* TITLE: sg_match - does object meet Star/Galaxy index criteria? */
/*          ie, min_sg <= avg(sg) <= max_sg */ 
/*                    input min,max sg index values */
/*                          number of SG values to average */
/*                          array of SG values */
/*                    return 1 = satisfies criteria */
/*                           0 = does NOT satisfies criteria */
int sg_match ( int minsg, int maxsg, int nvals, int *sg )
{
  double v1;
  int    i, nv;

  for (i = 0, v1 = 0.0, nv = 0; i < nvals; i++) {
    if (sg[i] != 19) {
      v1 += (double)sg[i];
      nv ++;
    }
  }
  if (nv > 0) v1 = v1/(double)nv;
  else        v1 = (double)minsg - 1; /* no points, invalid! */

  return ( LUB(v1,(double)minsg,(double)maxsg) );
}
