/* Written by Stephen Levine @ the US Naval Observatory, Flagstaff */
/* Copyright USNO, all rights reserved */

#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <string.h>

#ifdef STANDALONE2
#include "./Sla_bit/slalib.h"
#include "./Sla_bit/slamac.h"
#else
#include "/usr/local/include/slalib.h"
#include "/usr/local/include/slamac.h"
#endif

/* MACROS */
#define	ABS(x)		(((x) < 0) ? -(x) : (x))
#define	MAX(a,b)	(((a) < (b)) ? (b) : (a))
#define	MIN(a,b)	(((a) > (b)) ? (b) : (a))

#define TINY 1e-6

#define PI	 3.141592653589793238462643e0
#define TWOPI    6.283185307179586476925286766559e0
#define DEG2RAD  (PI / 180.0)
#define RAD2DEG  (180.0 / PI)
#define HR2RAD   (PI / 12.0)	/* M_PI is in <math.h> */
#define RAD2AS   (RAD2DEG * 3600.)
#define AS2RAD   (DEG2RAD / 3600.)

/* TITLE: rd2ze (ra,dec)->(xi, eta) */
void rd2ze (double ra, double dec, double rac, double decc, 
	    double *xi, double *eta, int *j)
{
  slaDs2tp(ra,dec,rac,decc,xi,eta,j);
  if ( *j > 0 ) {		/* error condition */
    *xi  += TWOPI;		/*  add 2pi to both xi and eta */
    *eta += TWOPI;
  }
  return;
}

/* TITLE: ze2rd (xi,eta)->(ra, dec) */
void ze2rd (double xi, double eta, double rac, double decc, 
	    double *ra, double *dec)
{
  slaDtp2s(xi,eta,rac,decc,ra,dec);
  if ( ABS(*ra - rac) > DPI && (*ra > DPI)) *ra = *ra - D2PI;
  return;
}

/* TITLE: radmnx  - compute min max and center for ra and dec 
  -if ra crosses zero, try to set to +/- instead of 23h and 0h version
  also compute x, y box limits */
/* radmnx rac decc rawid decwid &mnra &mxra &mndec &mxdec */
int radmnx (double rac, double decc, double raw, double decw,
	    double *mnra, double *mxra, double *mndec, double *mxdec,
	    double *mnbxx, double *mxbxx, double *mnbxy, double *mxbxy)
{
  double dx, dy, xvc[8], yvc[8], rvc[8], dvc[8], tmp;
  int i;

  dx = raw/2.;
  dy = decw/2.;
  *mnbxx = -dx;
  *mxbxx =  dx;
  *mnbxy = -dy;
  *mxbxy =  dy;
  
  if ( decc < 0. ) {
    xvc[0] = -dx;   yvc[0] = -dy; 
    xvc[1] =  dx;   yvc[1] = -dy;  
    for (i = 0; i < 2; i++){
      ze2rd (xvc[i], yvc[i], rac, decc, &(rvc[i]), &(dvc[i]));
    }
    mnmx (rvc, 2, mnra,  mxra);
    
    if ( rac > 1.5*DPI ) {
      if ( *mnra < DPIBY2 && *mxra > 1.5*DPI ) {
	tmp = *mnra;
	*mnra = *mxra - D2PI;
	*mxra = tmp;
      }
    }
  } else {
    xvc[0] = -dx;  yvc[0] = dy; 
    xvc[1] =  dx;  yvc[1] = dy;  
    
    for (i = 0; i < 2; i++){
      ze2rd (xvc[i], yvc[i], rac, decc, &(rvc[i]), &(dvc[i]));
    }
    mnmx (rvc, 2, mnra,  mxra);
    
    if ( rac > 1.5*DPI ) {
      if ( *mnra < DPIBY2 && *mxra > 1.5*DPI ) {
	tmp = *mnra;
	*mnra = *mxra - D2PI;
	*mxra = tmp;
      }
    }
  }

  xvc[0] = -dx;   yvc[0] = -dy; 
  xvc[1] = 0;	  yvc[1] = -dy;
  xvc[2] = dx;	  yvc[2] = -dy;  
  xvc[3] = dx;	  yvc[3] = 0;  
  xvc[4] = dx;	  yvc[4] = dy;  
  xvc[5] = 0;	  yvc[5] = dy;   
  xvc[6] = -dx;	  yvc[6] = dy; 
  xvc[7] = -dx;   yvc[7] = 0; 

  for (i = 0; i < 8; i++){
    ze2rd (xvc[i], yvc[i], rac, decc, &(rvc[i]), &(dvc[i]));
  }
  mnmx (dvc, 8, mndec, mxdec);

  if ((decc + dy) > DPIBY2) {
    *mxdec = DPIBY2;
    *mnra  = 0.0;
    *mxra  = D2PI;
  } else if ((decc - dy) < -DPIBY2) {
    *mndec = -DPIBY2;
    *mnra  =  0.0;
    *mxra  =  D2PI;
  }

  return ( 0 );
}

/* TITLE: mnmx - return min and max of a vector */
int mnmx ( double *vec, int npts, double *_mn, double *_mx )
{
  int i;

  *_mn = 1e36;
  *_mx = -1e36;

  for (i = 0; i < npts; i++) {
    *_mn = MIN(*_mn, vec[i]);
    *_mx = MAX(*_mx, vec[i]);
  }

  return ( 0 );
}

